import { describe, it, expect } from "vitest";
import html, { htmlStringToElement, htmlStringToElements } from "../html";
describe("HTML Service", () => {
    describe("html tagged template", () => {
        it("should concatenate strings without values", () => {
            const result = html `<div><p>Hello</p></div>`;
            expect(result).toBe("<div><p>Hello</p></div>");
        });
        it("should interpolate values", () => {
            const title = "Welcome";
            const content = "Hello World";
            const result = html `
        <div>
          <h1>${title}</h1>
          <p>${content}</p>
        </div>
      `;
            expect(result).toContain("<h1>Welcome</h1>");
            expect(result).toContain("<p>Hello World</p>");
        });
        it("should handle numeric values", () => {
            const count = 42;
            const result = html `<span>${count}</span>`;
            expect(result).toBe("<span>42</span>");
        });
        it("should handle multiple interpolations", () => {
            const id = "test-id";
            const className = "btn btn-primary";
            const text = "Click me";
            const result = html `<button id="${id}" class="${className}">
        ${text}
      </button>`;
            expect(result).toContain('id="test-id"');
            expect(result).toContain('class="btn btn-primary"');
            expect(result).toContain("Click me");
        });
        it("should preserve whitespace", () => {
            const result = html `
        <div>
          <span>Test</span>
        </div>
      `;
            expect(result).toContain("\n");
            expect(result).toContain("  ");
        });
        it("should handle empty template", () => {
            const result = html ``;
            expect(result).toBe("");
        });
        it("should convert undefined to empty string", () => {
            const value = undefined;
            const result = html `<div>${value}</div>`;
            expect(result).toBe("<div>undefined</div>");
        });
        it("should convert null to empty string", async () => {
            const value = null;
            const result = html `<div>${value}</div>`;
            expect(result).toBe("<div>null</div>");
        });
        it("should handle boolean values", () => {
            const enabled = true;
            const result = html `<input data-enabled="${enabled}" />`;
            expect(result).toBe('<input data-enabled="true" />');
        });
    });
    describe("htmlStringToElement", () => {
        it("should convert HTML string to single element", () => {
            const element = htmlStringToElement("<div>Hello</div>");
            expect(element).toBeInstanceOf(Element);
            expect(element.tagName).toBe("DIV");
            expect(element.textContent).toBe("Hello");
        });
        it("should handle elements with attributes", () => {
            const element = htmlStringToElement('<span id="test" class="active">Content</span>');
            expect(element.tagName).toBe("SPAN");
            expect(element.id).toBe("test");
            expect(element.className).toBe("active");
            expect(element.textContent).toBe("Content");
        });
        it("should handle nested elements", () => {
            const element = htmlStringToElement(`
        <div class="container">
          <h1>Title</h1>
          <p>Paragraph</p>
        </div>
      `);
            expect(element.tagName).toBe("DIV");
            expect(element.querySelector("h1")?.textContent).toBe("Title");
            expect(element.querySelector("p")?.textContent).toBe("Paragraph");
        });
        it("should trim whitespace from input", () => {
            const element = htmlStringToElement("  <div>Test</div>  ");
            expect(element).toBeInstanceOf(Element);
            expect(element.textContent).toBe("Test");
        });
        it("should handle self-closing elements", () => {
            const element = htmlStringToElement('<input type="text" />');
            expect(element.tagName).toBe("INPUT");
            expect(element.type).toBe("text");
        });
        it("should handle elements with data attributes", () => {
            const element = htmlStringToElement('<div data-test="value" data-number="42">Content</div>');
            expect(element.dataset.test).toBe("value");
            expect(element.dataset.number).toBe("42");
        });
        it("should return first child from template", () => {
            const element = htmlStringToElement("<p>First</p><p>Second</p>");
            expect(element.tagName).toBe("P");
            expect(element.textContent).toBe("First");
        });
    });
    describe("htmlStringToElements", () => {
        it("should convert HTML string to multiple elements", () => {
            const elements = htmlStringToElements("<div>First</div><span>Second</span>");
            expect(elements.length).toBe(2);
            expect(elements[0].textContent).toBe("First");
            expect(elements[1].textContent).toBe("Second");
        });
        it("should return NodeList", () => {
            const elements = htmlStringToElements("<div></div>");
            expect(elements).toHaveProperty("length");
            expect(elements).toHaveProperty("forEach");
        });
        it("should handle single element", () => {
            const elements = htmlStringToElements("<div>Single</div>");
            expect(elements.length).toBe(1);
            expect(elements[0].textContent).toBe("Single");
        });
        it("should handle empty string", () => {
            const elements = htmlStringToElements("");
            expect(elements.length).toBe(0);
        });
        it("should handle whitespace and text nodes", () => {
            const elements = htmlStringToElements("  <div>Test</div>  ");
            expect(elements.length).toBeGreaterThanOrEqual(1);
        });
        it("should preserve order of elements", () => {
            const elements = htmlStringToElements(`
        <div>1</div>
        <div>2</div>
        <div>3</div>
      `);
            const divs = Array.from(elements).filter((node) => node.nodeType === Node.ELEMENT_NODE);
            expect(divs[0].textContent?.trim()).toBe("1");
            expect(divs[1].textContent?.trim()).toBe("2");
            expect(divs[2].textContent?.trim()).toBe("3");
        });
        it("should handle complex nested structure", () => {
            const elements = htmlStringToElements(`
        <article>
          <header><h1>Title</h1></header>
          <section><p>Content</p></section>
        </article>
        <footer>Footer</footer>
      `);
            expect(elements.length).toBeGreaterThan(0);
            const article = Array.from(elements).find((node) => node.tagName === "ARTICLE");
            expect(article).toBeDefined();
            expect(article.querySelector("h1")?.textContent).toBe("Title");
        });
        it("should handle elements with various attributes", () => {
            const elements = htmlStringToElements(`
        <div id="first" class="item">1</div>
        <div id="second" class="item" data-value="test">2</div>
      `);
            const divs = Array.from(elements).filter((node) => node.tagName === "DIV");
            expect(divs[0].id).toBe("first");
            expect(divs[1].id).toBe("second");
            expect(divs[1].dataset.value).toBe("test");
        });
    });
});
