import { describe, it, expect } from "vitest";
import css from "../css";
describe("CSS Service", () => {
    it("should concatenate strings without values", () => {
        const result = css `
      .container {
        display: flex;
      }
    `;
        expect(result).toContain(".container");
        expect(result).toContain("display: flex");
    });
    it("should interpolate values", () => {
        const color = "red";
        const size = "16px";
        const result = css `
      .button {
        color: ${color};
        font-size: ${size};
      }
    `;
        expect(result).toContain("color: red");
        expect(result).toContain("font-size: 16px");
    });
    it("should handle numeric values", () => {
        const width = 100;
        const height = 200;
        const result = css `
      .box {
        width: ${width}px;
        height: ${height}px;
      }
    `;
        expect(result).toContain("width: 100px");
        expect(result).toContain("height: 200px");
    });
    it("should handle multiple interpolations", () => {
        const primary = "#007bff";
        const secondary = "#6c757d";
        const tertiary = "#28a745";
        const result = css `
      :root {
        --primary: ${primary};
        --secondary: ${secondary};
        --tertiary: ${tertiary};
      }
    `;
        expect(result).toContain("--primary: #007bff");
        expect(result).toContain("--secondary: #6c757d");
        expect(result).toContain("--tertiary: #28a745");
    });
    it("should preserve whitespace and formatting", () => {
        const result = css `
      .class {
        property: value;
      }
    `;
        expect(result).toContain("\n");
        expect(result).toContain("  ");
    });
    it("should handle empty template", () => {
        const result = css ``;
        expect(result).toBe("");
    });
    it("should convert undefined to empty string", () => {
        const value = undefined;
        const result = css `
      color: ${value};
    `;
        expect(result).toContain("color: undefined;");
    });
    it("should convert null to empty string", () => {
        const value = null;
        const result = css `
      color: ${value};
    `;
        expect(result).toContain("color: null;");
    });
    it("should handle boolean values", () => {
        const enabled = true;
        const result = css `
      enabled: ${enabled};
    `;
        expect(result).toContain("enabled: true;");
    });
    it("should handle complex nested CSS", () => {
        const primary = "blue";
        const hover = "darkblue";
        const result = css `
      .button {
        background: ${primary};
      }
      .button:hover {
        background: ${hover};
      }
    `;
        expect(result).toContain("background: blue");
        expect(result).toContain("background: darkblue");
    });
});
