import { describe, it, expect, vi, beforeEach } from "vitest";
import { cleanupZone, closeDetailsElements, cleanupTagsAndPreserveContent, cleanupOldSvgAttributes, randomizeSvgSpriteCalls, cleanupRandomSvgSpriteCallParams, } from "../cleanup";
vi.mock("./util", () => ({
    isLocalhost: vi.fn(() => true),
}));
describe("Cleanup Service", () => {
    describe("cleanupZone", () => {
        it("should remove elements with data-eden-remove-on-save attribute", () => {
            const html = `
        <div>
          <p>Keep this</p>
          <div data-eden-remove-on-save>Remove this</div>
          <span>Keep this too</span>
        </div>
      `;
            const result = cleanupZone(html);
            expect(result).toContain("Keep this");
            expect(result).toContain("Keep this too");
            expect(result).not.toContain("Remove this");
            expect(result).not.toContain("data-eden-remove-on-save");
        });
        it("should remove multiple elements with data-eden-remove-on-save", () => {
            const html = `
        <div>
          <div data-eden-remove-on-save>Remove 1</div>
          <div data-eden-remove-on-save>Remove 2</div>
          <p>Keep this</p>
        </div>
      `;
            const result = cleanupZone(html);
            expect(result).not.toContain("Remove 1");
            expect(result).not.toContain("Remove 2");
            expect(result).toContain("Keep this");
        });
        it("should handle empty HTML", () => {
            const result = cleanupZone("");
            expect(result).toBe("");
        });
        it("should handle HTML without elements to remove", () => {
            const html = "<div><p>Content</p></div>";
            const result = cleanupZone(html);
            expect(result).toContain("Content");
        });
    });
    describe("closeDetailsElements", () => {
        beforeEach(() => {
            document.body.innerHTML = "";
        });
        it("should close all open details elements in eden zones", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <details open>
            <summary>FAQ 1</summary>
            <p>Answer 1</p>
          </details>
          <details open>
            <summary>FAQ 2</summary>
            <p>Answer 2</p>
          </details>
        </div>
      `;
            closeDetailsElements();
            const details = document.querySelectorAll("details");
            details.forEach((detail) => {
                expect(detail.hasAttribute("open")).toBe(false);
            });
        });
        it("should only affect details inside eden zones", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <details open id="inside">
            <summary>Inside</summary>
          </details>
        </div>
        <details open id="outside">
          <summary>Outside</summary>
        </details>
      `;
            closeDetailsElements();
            const inside = document.getElementById("inside");
            const outside = document.getElementById("outside");
            expect(inside?.hasAttribute("open")).toBe(false);
            expect(outside?.hasAttribute("open")).toBe(true);
        });
        it("should not throw if no details elements exist", () => {
            document.body.innerHTML = "<div data-eden-zone></div>";
            expect(() => closeDetailsElements()).not.toThrow();
        });
    });
    describe("cleanupTagsAndPreserveContent", () => {
        beforeEach(() => {
            document.body.innerHTML = "";
        });
        it("should replace matched tags with their text content", () => {
            document.body.innerHTML = `
        <div>
          <span class="remove-me">Text to preserve</span>
        </div>
      `;
            cleanupTagsAndPreserveContent(".remove-me");
            expect(document.body.innerHTML).toContain("Text to preserve");
            expect(document.body.innerHTML).not.toContain("span");
            expect(document.body.innerHTML).not.toContain("remove-me");
        });
        it("should handle multiple matching elements", () => {
            document.body.innerHTML = `
        <div>
          <span class="cleanup">Text 1</span>
          <span class="cleanup">Text 2</span>
          <span class="cleanup">Text 3</span>
        </div>
      `;
            cleanupTagsAndPreserveContent(".cleanup");
            expect(document.body.innerHTML).toContain("Text 1");
            expect(document.body.innerHTML).toContain("Text 2");
            expect(document.body.innerHTML).toContain("Text 3");
        });
        it("should skip elements without text content", () => {
            document.body.innerHTML = `
        <div>
          <span class="empty"></span>
          <span class="empty">Content</span>
        </div>
      `;
            cleanupTagsAndPreserveContent(".empty");
            const spans = document.querySelectorAll("span");
            expect(spans.length).toBeGreaterThan(0);
        });
        it("should not throw with invalid selector", () => {
            document.body.innerHTML = "<div></div>";
            expect(() => cleanupTagsAndPreserveContent(".nonexistent")).not.toThrow();
        });
    });
    describe("cleanupOldSvgAttributes", () => {
        beforeEach(() => {
            document.body.innerHTML = "";
        });
        it("should replace xlink:href with href on SVG use elements", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use xlink:href="#icon-test"></use>
          </svg>
        </div>
      `;
            cleanupOldSvgAttributes();
            const useElement = document.querySelector("use");
            expect(useElement?.hasAttribute("xlink:href")).toBe(false);
            expect(useElement?.getAttribute("href")).toBe("#icon-test");
        });
        it("should handle multiple SVG use elements", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use xlink:href="#icon1"></use>
            <use xlink:href="#icon2"></use>
          </svg>
        </div>
      `;
            cleanupOldSvgAttributes();
            const useElements = document.querySelectorAll("use");
            useElements.forEach((use) => {
                expect(use.hasAttribute("xlink:href")).toBe(false);
                expect(use.hasAttribute("href")).toBe(true);
            });
        });
        it("should not modify use elements without xlink:href", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use href="#icon-modern"></use>
          </svg>
        </div>
      `;
            cleanupOldSvgAttributes();
            const useElement = document.querySelector("use");
            expect(useElement?.getAttribute("href")).toBe("#icon-modern");
        });
        it("should handle empty xlink:href value", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use xlink:href=""></use>
          </svg>
        </div>
      `;
            cleanupOldSvgAttributes();
            const useElement = document.querySelector("use");
            expect(useElement?.hasAttribute("xlink:href")).toBe(true);
        });
    });
    describe("randomizeSvgSpriteCalls", () => {
        beforeEach(() => {
            document.body.innerHTML = "";
            window.location = {
                origin: "http://localhost:3000",
                hostname: "localhost",
            };
        });
        it("should add random parameter to SVG sprite URLs", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use href="/sprites.svg#icon"></use>
          </svg>
        </div>
      `;
            randomizeSvgSpriteCalls();
            const useElement = document.querySelector("use");
            const href = useElement?.getAttribute("href");
            expect(href).toContain("sprites.svg");
            expect(href).toContain("?r=");
            expect(href).toContain("#icon");
        });
        it("should preserve hash in URL", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use href="/sprites.svg#my-icon"></use>
          </svg>
        </div>
      `;
            randomizeSvgSpriteCalls();
            const useElement = document.querySelector("use");
            const href = useElement?.getAttribute("href");
            expect(href).toContain("#my-icon");
        });
        it("should handle xlink:href attribute", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use xlink:href="/sprites.svg#icon"></use>
          </svg>
        </div>
      `;
            randomizeSvgSpriteCalls();
            const useElement = document.querySelector("use");
            const href = useElement?.getAttribute("href");
            expect(href).toBeTruthy();
            expect(href).toContain("sprites.svg");
        });
    });
    describe("cleanupRandomSvgSpriteCallParams", () => {
        beforeEach(() => {
            document.body.innerHTML = "";
            window.EdenConfig = { lang: "en" };
            window.location = {
                origin: "http://localhost:3000",
                hostname: "localhost",
            };
        });
        it("should remove random parameter from SVG sprite URLs", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use href="/en/sprites.svg?r=5#icon"></use>
          </svg>
        </div>
      `;
            cleanupRandomSvgSpriteCallParams();
            const useElement = document.querySelector("use");
            const href = useElement?.getAttribute("href");
            expect(href).toBe("/sprites.svg#icon");
        });
        it("should remove language prefix from path", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use href="/en/sprites.svg?r=3#icon"></use>
          </svg>
        </div>
      `;
            cleanupRandomSvgSpriteCallParams();
            const useElement = document.querySelector("use");
            const href = useElement?.getAttribute("href");
            expect(href).toBe("/sprites.svg#icon");
        });
        it("should preserve hash in URL", () => {
            document.body.innerHTML = `
        <div data-eden-zone>
          <svg>
            <use href="/en/sprites.svg?r=7#my-icon"></use>
          </svg>
        </div>
      `;
            cleanupRandomSvgSpriteCallParams();
            const useElement = document.querySelector("use");
            const href = useElement?.getAttribute("href");
            expect(href).toContain("#my-icon");
        });
    });
});
