import Block, { pasteFromClipboard, clipboardError } from "./block.js";
import { snackbar } from "../user-interface/common/eden-snackbar.js";
import { fireEvent } from "../services/event.js";
import { isLocalhost } from "../services/util.js";
import buildUuid from "../services/uuid.js";
import t from "../i18n/index.js";
export default class Zone {
    constructor(element, templateDescriptors) {
        this.analyse = async () => {
            const zone = this.element;
            const domBlocks = Array.from(zone.querySelectorAll("section"));
            const blocks = (await Promise.all(domBlocks.map(async (domBlock) => {
                const templateName = domBlock.getAttribute("data-template");
                const templateDescriptor = this.templateDescriptors?.[templateName];
                if (!templateDescriptor || !templateName)
                    return;
                const block = new Block(domBlock, templateDescriptor, this, templateName, false);
                try {
                    await block.init();
                }
                catch (e) {
                    if (isLocalhost()) {
                        console.info(e);
                    }
                }
                return block;
            }))).filter((block) => Boolean(block));
            if (this.defaultContent && !this.defaultContentSpawned && !blocks.length) {
                await this.create(this.defaultContent, undefined, false);
                await this.analyse();
                this.defaultContentSpawned = true;
                return;
            }
            this.blocks = blocks;
        };
        this.cleanRefs = () => {
            if (!this.blocks)
                return;
            this.blocks.forEach((block) => block.cleanRefs());
            delete this.blocks;
            delete this.templateDescriptors;
        };
        this.showUi = async () => {
            if (!this.blocks)
                return;
            this.hideUi();
            const isEmpty = this.blocks.length === 0;
            if (!this.isReadOnly && isEmpty) {
                const { default: EdenZoneUi } = await import("../user-interface/zone/eden-zone-ui.js");
                this.ui = new EdenZoneUi(this, {
                    addBlock: this.create,
                    paste: this.paste,
                }, this.templates, this.maxBlocks);
                this.element.appendChild(this.ui);
            }
        };
        this.hideUi = () => {
            if (this.ui !== undefined) {
                this.ui.remove();
            }
        };
        this.activate = () => {
            if (!this.blocks)
                return;
            this.showUi();
            this.blocks.forEach((block) => block.activate());
        };
        this.desactivate = () => {
            if (!this.blocks)
                return;
            this.hideUi();
            this.blocks.forEach((block) => block.desactivate());
        };
        this.preSave = () => {
            if (!this.blocks)
                return;
            this.blocks.forEach((block) => block.preSave());
        };
        this.postSave = () => {
            if (!this.blocks)
                return;
            this.blocks.forEach((block) => block.postSave());
        };
        this.displayInfo = () => {
            if (!this.blocks)
                return;
            this.blocks.forEach((block) => block.displayInfo());
        };
        this.hideInfo = () => {
            if (!this.blocks)
                return;
            this.blocks.forEach((block) => block.hideInfo());
        };
        this.addUuids = () => {
            if (!this.withUuid)
                return;
            const blocks = Array.from(this.element.querySelectorAll("section[data-template]"));
            blocks.forEach((block) => {
                const uuid = buildUuid();
                block.setAttribute("data-uuid", uuid);
            });
        };
        this.paste = async () => {
            if (navigator && navigator.clipboard) {
                try {
                    await pasteFromClipboard("afterbegin", this.element);
                    this.addUuids();
                    this.desactivate();
                    await this.analyse();
                    this.activate();
                    this.hideUi();
                    snackbar(t("snackbars.pasted-content"), "success");
                }
                catch (e) {
                    if (isLocalhost()) {
                        console.info("Reading from clipboard error", e);
                    }
                    snackbar(t("snackbars.paste-error"), "error");
                }
            }
            else {
                clipboardError();
            }
        };
        this.create = async (code, variant, showBlockUi = true) => {
            const block = await this.createBlock(code, variant, showBlockUi);
            if (block) {
                this.element.append(block.element);
                if (showBlockUi) {
                    block.showUi();
                }
                this.showUi();
                if (block.zone) {
                    fireEvent(document, "eden-block-created", {
                        block,
                        zone: this,
                    });
                }
            }
        };
        this.createBlock = async (templateCode, variant, activateBlock = true) => {
            if (this.canAddBlock()) {
                const templateDescriptor = this.templateDescriptors?.[templateCode];
                if (!templateDescriptor)
                    return;
                const element = templateDescriptor.createElement(templateCode, this, variant);
                const block = new Block(element, templateDescriptor, this, templateCode, true);
                await block.init();
                this.blocks?.push(block);
                if (activateBlock) {
                    block.activate();
                }
                return block;
            }
        };
        this.copy = async () => {
            if (navigator && navigator.clipboard) {
                try {
                    this.desactivate();
                    const copy = this.element.cloneNode(true);
                    this.activate();
                    await navigator.clipboard.writeText(`fromEden:${copy.innerHTML}`);
                    snackbar(t("snackbars.copied-blocks"), "success");
                }
                catch (e) {
                    if (isLocalhost()) {
                        console.info("writing into clipboard error", e);
                    }
                    snackbar(t("snackbars.copy-error"), "error");
                }
            }
            else {
                clipboardError();
            }
        };
        this.canAddBlock = () => {
            if (!this.blocks)
                return true;
            return this.maxBlocks === undefined || this.blocks.length < this.maxBlocks;
        };
        this.element = element;
        const { dataset } = this.element;
        this.defaultContent = dataset.edenDefaultContent;
        this.maxBlocks = dataset.edenMaxBlocks
            ? parseInt(dataset.edenMaxBlocks, 10)
            : undefined;
        const templates = dataset.edenTemplates;
        this.templates = templates ? templates.split(",") : undefined;
        this.tagsLabel = dataset.edenTagsLabel;
        this.allowedTags = dataset.edenAllowedTags;
        this.allowMultipleTags = dataset.edenAllowMultipleTags === "true";
        this.isFixed = dataset.edenFixed === "true";
        this.isReadOnly = dataset.edenReadOnly === "true";
        this.withUuid = dataset.edenUuid === "true";
        this.hasChanged = false;
        this.defaultContentSpawned = false;
        this.templateDescriptors = templateDescriptors;
        this.blocks = [];
    }
}
