import EdenModal from "../user-interface/common/eden-modal.js";
import { snackbar } from "../user-interface/common/eden-snackbar.js";
import Template from "./template.js";
import { isLocalhost, smoothScrollTo } from "../services/util.js";
import typeOf from "../services/type-of.js";
import { fireEvent } from "../services/event.js";
import buildUuid from "../services/uuid.js";
import Text from "../bricks/Text.js";
import RichText from "../bricks/RichText.js";
import Picture from "../bricks/Picture.js";
import Link from "../bricks/Link.js";
import Icon from "../bricks/Icon.js";
import Slideshow from "../bricks/Slideshow.js";
import Table from "../bricks/Table.js";
import Recursive from "../bricks/Recursive.js";
import Video from "../bricks/Video.js";
import Iframe from "../bricks/Iframe.js";
import GoogleMaps from "../bricks/GoogleMaps.js";
import YoutubePlaylist from "../bricks/YoutubePlaylist.js";
import Spotify from "../bricks/Spotify.js";
import t from "../i18n/index.js";
import { generateColor } from "../services/colors.js";
const Templates = {
    Text,
    RichText,
    Picture,
    Link,
    Icon,
    Slideshow,
    Table,
    Recursive,
    Video,
    Iframe,
    GoogleMaps,
    YoutubePlaylist,
    Spotify,
};
export default class Block {
    constructor(element, template, zone, code, isCreation) {
        this.init = () => {
            if (!this.template)
                return;
            try {
                this.element.addEventListener("mouseenter", this.onMouseEnter);
                this.element.addEventListener("mouseleave", this.onMouseLeave);
                this.checkDates();
                this.setupRemoteEventListeners();
                return recursInit(this, this.template.templates, Boolean(this.isCreation));
            }
            catch {
                if (isLocalhost()) {
                    console.info("[eden] brick not found: " + this.code);
                }
            }
        };
        this.setupRemoteEventListeners = () => {
            this.remoteContentUpdateHandler = this.handleRemoteContentUpdate.bind(this);
            this.remoteOperationHandler = this.handleRemoteOperation.bind(this);
            this.remoteSelectionHandler = this.handleRemoteSelection.bind(this);
            document.addEventListener("eden-block-content-received", this.remoteContentUpdateHandler);
            document.addEventListener("eden-block-operation-received", this.remoteOperationHandler);
            document.addEventListener("eden-block-selections-updated", this.remoteSelectionHandler);
        };
        this.handleRemoteContentUpdate = async (event) => {
            const customEvent = event;
            const { update } = customEvent.detail;
            if (!this.zone || !update)
                return;
            const zoneId = window.Eden.zones.indexOf(this.zone);
            const blockId = this.zone.blocks?.indexOf(this);
            if (zoneId === update.zoneId && blockId === update.blockId) {
                this.element.innerHTML = update.content;
                const isConnected = document.documentElement.getAttribute("data-eden-mode") === "connected";
                if (!isConnected) {
                    this.zone.desactivate();
                }
                await this.zone.analyse();
                if (!isConnected) {
                    this.zone.activate();
                }
            }
        };
        this.handleRemoteOperation = async (event) => {
            const customEvent = event;
            const { operation } = customEvent.detail;
            if (!this.zone || !operation)
                return;
            const zoneId = window.Eden.zones.indexOf(this.zone);
            const blockId = this.zone.blocks?.indexOf(this);
            if (zoneId === operation.zoneId && blockId === operation.blockId) {
                if (operation.operation === "removed") {
                    this.element.remove();
                    this.zone.blocks =
                        this.zone.blocks?.filter((block) => block !== this) || [];
                }
                if (operation.operation === "added") {
                }
            }
        };
        this.handleRemoteSelection = async (event) => {
            const customEvent = event;
            const { selections } = customEvent.detail;
            if (!this.zone || !selections)
                return;
            const zoneId = window.Eden.zones.indexOf(this.zone);
            const blockId = this.zone.blocks?.indexOf(this);
            const remoteSelection = selections.find((sel) => sel.zoneId === zoneId && sel.blockId === blockId);
            if (remoteSelection) {
                this.showRemoteSelectionIndicator(remoteSelection);
                this.isLockedByRemotePlayer = true;
            }
            else {
                this.hideRemoteSelectionIndicator();
                this.isLockedByRemotePlayer = false;
            }
        };
        this.showRemoteSelectionIndicator = async (selection) => {
            if (this.selectionIndicator)
                return;
            const { default: EdenPlayerSelectionIndicator } = await import("../user-interface/block/eden-player-selection-indicator.js");
            const zoneId = this.zone ? window.Eden.zones.indexOf(this.zone) : -1;
            const blockId = this.zone?.blocks?.indexOf(this) ?? -1;
            this.selectionIndicator = new EdenPlayerSelectionIndicator({
                clientId: selection.clientId,
                firstName: selection.firstName,
                lastName: selection.lastName,
                zoneId,
                blockId,
            });
            const initials = `${selection.firstName.charAt(0)}${selection.lastName.charAt(0)}`.toUpperCase();
            const color = generateColor(initials);
            this.selectionIndicator.style.setProperty("--player-color", `var(${color})`);
            if (this.initialPosition !== "relative") {
                this.element.style.position = "relative";
            }
            this.element.appendChild(this.selectionIndicator);
        };
        this.hideRemoteSelectionIndicator = () => {
            if (this.selectionIndicator) {
                this.selectionIndicator.remove();
                this.selectionIndicator = undefined;
                if (this.initialPosition && this.initialPosition !== "relative") {
                    this.element.style.position = this.initialPosition;
                }
            }
        };
        this.activate = () => {
            this.element.addEventListener("click", this.onClick);
            this.element.addEventListener("paste", this.handlePaste);
            this.observer = new MutationObserver((entries) => {
                this.contentHasChanged = true;
                this.observer?.disconnect();
            });
            this.observer.observe(this.element, {
                characterData: true,
                attributes: true,
                childList: true,
                subtree: true,
            });
            this.instances?.forEach((instance) => {
                if (typeof instance === "object") {
                    instance.activate();
                }
            });
        };
        this.desactivate = () => {
            this.element.removeEventListener("paste", this.handlePaste);
            this.observer?.disconnect();
            this.hideUi();
            this.instances?.forEach((instance) => instance.desactivate());
        };
        this.cleanRefs = () => {
            this.instances?.forEach((instance) => instance.cleanRefs());
            this.element.removeEventListener("mouseleave", this.onMouseLeave);
            this.element.removeEventListener("mouseenter", this.onMouseEnter);
            this.element.removeEventListener("mousemove", this.onMouseMove);
            if (this.remoteContentUpdateHandler) {
                document.removeEventListener("eden-block-content-received", this.remoteContentUpdateHandler);
            }
            if (this.remoteOperationHandler) {
                document.removeEventListener("eden-block-operation-received", this.remoteOperationHandler);
            }
            if (this.remoteSelectionHandler) {
                document.removeEventListener("eden-block-selections-updated", this.remoteSelectionHandler);
            }
            delete this.instances;
            delete this.template;
            delete this.zone;
        };
        this.preSave = () => {
            this.cleanupDatesCheck();
            this.instances?.forEach((instance) => {
                if (typeof instance === "object") {
                    instance.preSave();
                }
            });
        };
        this.postSave = () => {
            this.instances?.forEach((instance) => {
                if (typeof instance === "object") {
                    instance.postSave();
                }
            });
            this.checkDates();
        };
        this.displayInfo = async () => {
            const { dates, hidden } = this.element.dataset;
            const { default: EdenInfoUi } = await import("../user-interface/block/eden-info-ui.js");
            this.infoUi = new EdenInfoUi(this.element, dates, hidden);
            this.element.appendChild(this.infoUi);
            if (this.initialPosition !== "relative") {
                this.element.style.position = "relative";
            }
        };
        this.hideInfo = () => {
            if (this.infoUi) {
                this.infoUi.remove();
                delete this.infoUi;
                if (this.initialPosition && this.initialPosition !== "relative") {
                    this.element.style.position = this.initialPosition;
                }
            }
        };
        this.create = (position) => async (code, variant) => {
            if (!this.zone)
                return;
            const block = await this.zone.createBlock(code, variant);
            if (block) {
                if ("startViewTransition" in document) {
                    setViewTransitionStyles();
                    document.startViewTransition(() => {
                        this.element.insertAdjacentElement(position, block.element);
                        block.showUi();
                        const delay = getViewTransitionDuration() + 40;
                        setTimeout(() => {
                            smoothScrollTo(block.element);
                        }, delay);
                    });
                    removeViewTransitionStyles();
                }
                else {
                    this.element.insertAdjacentElement(position, block.element);
                    block.showUi();
                    smoothScrollTo(block.element);
                }
                if (block.zone) {
                    fireEvent(document, "eden-block-created", {
                        block,
                        zone: block.zone,
                    });
                }
            }
        };
        this.createBefore = this.create("beforebegin");
        this.createAfter = this.create("afterend");
        this.remove = () => {
            if (!this.zone)
                return;
            fireEvent(document, "eden-block-removed", {
                block: this,
                zone: this.zone,
            });
            if ("startViewTransition" in document) {
                setViewTransitionStyles();
                document.startViewTransition(() => {
                    this.element.remove();
                });
                removeViewTransitionStyles();
            }
            else {
                this.element.remove();
            }
            this.zone.blocks =
                this.zone.blocks?.filter((block) => block !== this) || [];
            this.zone.showUi();
        };
        this.copy = async (elementType, cutMode) => {
            this.zone?.desactivate();
            const copy = (elementType === "zone" && this.zone
                ? this.zone.element.cloneNode(true)
                : this.element.cloneNode(true));
            if (cutMode) {
                this.remove();
            }
            this.zone?.activate();
            await navigator.clipboard.writeText(`fromEden:${elementType === "zone" ? copy.innerHTML : copy.outerHTML}`);
        };
        this.manualCopy = (elementType, cutMode) => async () => {
            if (navigator && navigator.clipboard) {
                try {
                    let message = "snackbars.copied-blocks";
                    if (elementType === "block") {
                        message = cutMode
                            ? "snackbars.cut-block"
                            : "snackbars.copied-block";
                    }
                    await this.copy(elementType, cutMode);
                    snackbar(t(message), "success");
                }
                catch (e) {
                    if (isLocalhost()) {
                        console.info("writing into clipboard error", e);
                    }
                    snackbar(t("snackbars.copy-error"), "error");
                }
            }
            else {
                clipboardError();
            }
        };
        this.copyZone = this.manualCopy("zone");
        this.copyBlock = this.manualCopy("block");
        this.cutBlock = this.manualCopy("block", true);
        this.copyDates = async () => {
            const { dates } = this.element.dataset;
            if (dates) {
                window.sessionStorage.setItem("copiedDates", dates);
                snackbar(t("snackbars.copied-dates"), "success");
            }
            else {
                snackbar(t("snackbars.no-dates-to-copy"), "warning");
            }
        };
        this.setOrUpdateUuids = () => {
            if (!this.zone?.withUuid)
                return;
            const blocks = Array.from(this.zone.element.querySelectorAll("section[data-template]")).reverse();
            blocks.forEach((block) => {
                const currentUuid = block.dataset.uuid;
                const isDuplicate = blocks.filter((s) => s !== block && s.dataset.uuid === currentUuid)
                    .length > 0;
                if (isDuplicate || !currentUuid) {
                    const uuid = buildUuid();
                    block.dataset.uuid = uuid;
                }
            });
        };
        this.paste = async (position) => {
            if (!this.zone)
                return;
            await pasteFromClipboard(position, this.element);
            this.setOrUpdateUuids();
            this.zone.desactivate();
            await this.zone.analyse();
            this.zone.activate();
            fireEvent(document, "eden-block-pasted", {
                block: this,
                zone: this.zone,
            });
        };
        this.manualPaste = (position) => async () => {
            if (navigator && navigator.clipboard) {
                try {
                    await this.paste(position);
                    snackbar(t("snackbars.pasted-content"), "success");
                }
                catch (e) {
                    if (isLocalhost()) {
                        console.info("Reading from clipboard error", e);
                    }
                    snackbar(t("snackbars.paste-error"), "error");
                }
            }
            else {
                clipboardError();
            }
        };
        this.pasteBefore = this.manualPaste("beforebegin");
        this.pasteAfter = this.manualPaste("afterend");
        this.duplicate = async () => {
            if (navigator && navigator.clipboard) {
                await this.copy("block");
                await this.paste("beforebegin");
            }
            else {
                clipboardError();
            }
        };
        this.selectDates = async () => {
            const { timeProgrammableBlocks } = window.EdenConfig;
            if (!timeProgrammableBlocks)
                return;
            const selectedDates = this.element.dataset.dates;
            const copiedDates = window.sessionStorage.getItem("copiedDates");
            const { default: SelectDatesModal } = await import("../user-interface/plugin/select-dates-modal.js");
            new SelectDatesModal(this.changeDates, copiedDates, selectedDates).init();
        };
        this.settings = async () => {
            if (!this.zone)
                return;
            const { default: SettingsModal } = await import("../user-interface/plugin/settings-modal.js");
            new SettingsModal(this.changeSettings, {
                id: this.element.id,
                tags: this.element.dataset.tags || "",
                hidden: this.element.dataset.hidden === "true" || false,
                variant: this.variant,
                tagsLabel: this.zone.tagsLabel,
                allowedTags: this.zone.allowedTags,
                allowMultipleTags: this.zone.allowMultipleTags,
                attributes: this.attributes,
            }, this.template?.variants, this.template?.defaultVariantName).init();
        };
        this.changeSettings = ({ id, tags, variant, hidden, attributes, }) => {
            if (id === "") {
                this.element.removeAttribute("id");
            }
            else {
                this.element.setAttribute("id", id);
            }
            if (tags === "") {
                this.element.removeAttribute("data-tags");
            }
            else {
                this.element.dataset.tags = tags;
            }
            if (!hidden) {
                this.element.removeAttribute("data-hidden");
                this.element.removeAttribute("data-nosnippet");
            }
            else {
                this.element.dataset.hidden = "true";
                this.element.setAttribute("data-nosnippet", "");
            }
            this.updateVariant(variant);
            if (attributes) {
                attributes.forEach(({ property, value }) => {
                    if (value) {
                        this.element.setAttribute(property, value);
                    }
                    else {
                        this.element.removeAttribute(property);
                    }
                });
            }
            this.attributes = this.template?.getAttributes(this.element);
        };
        this.updateVariant = (variant) => {
            if (this.variant) {
                this.element.classList.remove(this.variant);
            }
            if (variant && variant !== "") {
                this.element.classList.add(variant);
            }
            this.variant = this.template?.getVariant(this.element);
            if (this.ui)
                this.ui.variant = this.variant;
        };
        this.changeDates = (dates) => {
            if (dates === "[]") {
                delete this.element.dataset.dates;
            }
            else {
                this.element.dataset.dates = dates;
            }
            snackbar(t("snackbars.applied-dates"), "success");
        };
        this.move = (position) => () => {
            const sibling = position === "beforebegin"
                ? this.element.previousElementSibling
                : this.element.nextElementSibling;
            if (sibling) {
                this.hideUi();
                if ("startViewTransition" in document) {
                    setViewTransitionStyles();
                    document.startViewTransition(() => {
                        sibling.insertAdjacentElement(position, this.element);
                        this.showUi();
                        const delay = getViewTransitionDuration() + 40;
                        setTimeout(() => {
                            smoothScrollTo(this.element);
                        }, delay);
                    });
                    removeViewTransitionStyles();
                }
                else {
                    sibling.insertAdjacentElement(position, this.element);
                    this.showUi();
                    smoothScrollTo(this.element);
                }
            }
        };
        this.moveBefore = this.move("beforebegin");
        this.moveAfter = this.move("afterend");
        this.showUi = async (e) => {
            const activeBlock = window.Eden.getActiveBlock();
            if (this.isLockedByRemotePlayer) {
                if (e) {
                    e.preventDefault();
                    e.stopPropagation();
                }
                return;
            }
            if (this.zone && this.zone.isFixed) {
                this.zone.element.setAttribute("data-eden-modified", "true");
            }
            if (this.obsolete && e) {
                e.preventDefault();
            }
            if (activeBlock === this) {
                return;
            }
            else if (activeBlock !== undefined) {
                activeBlock.hideUi();
            }
            window.Eden.setActiveBlock(this);
            fireEvent(document, "eden-active-block-changed", { block: this });
            const disabledActions = [];
            if (!this.zone?.canAddBlock())
                disabledActions.push("add");
            if (this.isFirstChild())
                disabledActions.push("moveBefore");
            if (this.isLastChild())
                disabledActions.push("moveAfter");
            const { default: EdenBlockUi } = await import("../user-interface/block/eden-block-ui.js");
            this.ui = new EdenBlockUi(this, {
                addBefore: this.createBefore,
                addAfter: this.createAfter,
                remove: this.remove,
                copy: this.copyBlock,
                cut: this.cutBlock,
                copyZone: this.copyZone,
                copyDates: this.copyDates,
                pasteBefore: this.pasteBefore,
                pasteAfter: this.pasteAfter,
                moveBefore: this.moveBefore,
                moveAfter: this.moveAfter,
                duplicate: this.duplicate,
                selectDates: this.selectDates,
                settings: this.settings,
                updateVariant: this.updateVariant,
            }, disabledActions, this.zone?.templates, this.zone?.maxBlocks, this.obsolete, this.variant, this.template?.variants);
            if (this.initialPosition !== "relative") {
                this.element.style.position = "relative";
            }
            document.addEventListener("click", this.handleOutsideClick);
            patchDetails(this.element);
            this.addToolbars();
            this.element.appendChild(this.ui);
            window.Eden.zones.forEach((zone) => {
                if (zone.blocks && zone.blocks.length > 0) {
                    zone.hideUi();
                }
            });
        };
        this.hideUi = () => {
            const activeBlock = window.Eden.getActiveBlock();
            if (activeBlock === this) {
                window.Eden.setActiveBlock(undefined);
                this.ui?.remove();
                delete this.ui;
                this.activePlugins = [];
                if (this.initialPosition && this.initialPosition !== "relative") {
                    this.element.style.position = this.initialPosition;
                }
                cleanupDetailsPatch(this.element);
                this.element.style.removeProperty("z-index");
                document.removeEventListener("click", this.handleOutsideClick);
                fireEvent(document, "eden-active-block-changed", { block: undefined });
                if (this.contentHasChanged) {
                    fireEvent(document, "eden-block-content-changed", {
                        block: this,
                        content: this.element.innerHTML,
                    });
                }
            }
        };
        this.addActions = (actionsGroups) => {
            this.actionsGroups = [...(this.actionsGroups || []), ...actionsGroups];
            this.addToolbars();
        };
        this.addToolbars = () => {
            if (this.ui && this.actionsGroups && this.actionsGroups.length > 0) {
                this.ui.addToolbars(this.actionsGroups);
                this.actionsGroups = [];
            }
        };
        this.removeActions = (actionsGroups) => {
            if (this.ui && actionsGroups && actionsGroups.length > 0) {
                this.ui.removeActions(actionsGroups);
            }
        };
        this.manageActions = async (pluginsToActivate) => {
            await this.showUi();
            const toRemove = this.activePlugins?.filter((activePlugin) => !pluginsToActivate.includes(activePlugin));
            const toAdd = pluginsToActivate.filter((plugin) => !this.activePlugins?.includes(plugin));
            const actionsGroupsToRemove = toRemove?.map((plugin) => plugin.actions);
            this.removeActions(actionsGroupsToRemove);
            const actionsGroupsToAdd = toAdd.map((plugin) => ({
                toolbar: plugin.actions,
                watchSelectionChange: Boolean(plugin.watchSelectionChange),
            }));
            if (actionsGroupsToAdd) {
                this.addActions(actionsGroupsToAdd);
            }
            this.activePlugins = pluginsToActivate;
        };
        this.onClick = async (e) => {
            await this.showUi(e);
        };
        this.handleOutsideClick = (e) => {
            const target = e.target;
            const ignoreClick = target.closest("eden-icon-dropdown, eden-add-block, eden-dropdown, eden-modal");
            const isAnchorClick = target.closest("a");
            const activeBlock = window.Eden.getActiveBlock();
            const selection = window.getSelection();
            if (!selection)
                return;
            const isSelecting = selection.type === "Range";
            if (!isAnchorClick &&
                (ignoreClick ||
                    !activeBlock ||
                    activeBlock.element?.contains(target) ||
                    isSelecting)) {
                return;
            }
            this.hideUi();
            window.Eden.zones.forEach((zone) => zone.showUi());
        };
        this.showHelper = async (e) => {
            const currentMode = document.documentElement.getAttribute("data-eden-mode");
            if (currentMode !== "info") {
                return this.hideHelper();
            }
            const target = e.target;
            const block = target.closest("section");
            if (!block)
                return;
            const { clientX, clientY } = e;
            if (!this.helper) {
                const { default: EdenHelper } = await import("../user-interface/common/eden-helper.js");
                this.helper = new EdenHelper(block);
                document.body.appendChild(this.helper);
            }
            const helperWidth = this.helper.getBoundingClientRect().width;
            const hasEnoughPlace = clientX + helperWidth < window.innerWidth;
            let transform = `translate(${clientX + 14}px, ${clientY - 10}px)`;
            if (!hasEnoughPlace) {
                transform = `translate(${clientX - helperWidth - 14}px, ${clientY - 10}px)`;
            }
            this.helper.style.transform = transform;
        };
        this.hideHelper = () => {
            if (this.helper) {
                this.helper.remove();
                this.helper = undefined;
            }
        };
        this.onMouseEnter = () => {
            this.element.addEventListener("mousemove", this.onMouseMove);
        };
        this.onMouseLeave = () => {
            this.element.removeEventListener("mousemove", this.onMouseMove);
            this.hideHelper();
        };
        this.onMouseMove = (e) => {
            const currentMode = document.documentElement.getAttribute("data-eden-mode");
            if (currentMode === "info") {
                this.showHelper(e);
            }
            else {
                this.hideHelper();
            }
        };
        this.handlePaste = async (e) => {
            e.stopPropagation();
            e.preventDefault();
            const { clipboardData } = e || window;
            if (!clipboardData)
                return;
            const pastedData = cleanupClipboardData(clipboardData.getData("text/plain"));
            document.execCommand("insertHTML", false, pastedData);
        };
        this.isFirstChild = () => this.zone?.element.firstChild === this.element;
        this.isLastChild = () => this.zone?.element.lastChild === this.element;
        this.checkDates = () => {
            this.cleanupDatesCheck();
            if (!this.element.dataset.dates)
                return;
            const now = new Date();
            const dates = JSON.parse(this.element.dataset.dates);
            const active = dates.reduce((acc, [start, end]) => {
                const dateStart = new Date(start);
                const dateEnd = new Date(end);
                const isActive = dateStart <= now && now <= dateEnd;
                if ((acc === false && isActive) ||
                    (acc !== false && dateStart <= new Date(acc))) {
                    return end;
                }
                return acc;
            }, false);
            if (active) {
                this.element.dataset.datesActive = formateDate(active);
                return;
            }
            const scheduled = dates.reduce((acc, [start]) => {
                const dateStart = new Date(start);
                const isScheduled = now <= dateStart;
                if ((acc === false && isScheduled) ||
                    (acc !== false && dateStart <= new Date(acc))) {
                    return start;
                }
                return acc;
            }, false);
            if (scheduled) {
                this.element.dataset.datesScheduled = formateDate(scheduled);
            }
            else {
                this.element.dataset.datesScheduled = scheduled;
            }
        };
        this.cleanupDatesCheck = () => {
            delete this.element.dataset.datesActive;
            delete this.element.dataset.datesScheduled;
        };
        this.element = element;
        if (template) {
            this.template = new Template(template);
            this.variant = this.template.getVariant(this.element);
            this.attributes = this.template.getAttributes(this.element);
        }
        else {
            this.obsolete = true;
        }
        this.zone = zone;
        this.code = code;
        this.isCreation = isCreation;
        this.initialPosition = element.style.position;
        this.helper = undefined;
        this.instances = [];
        this.activePlugins = [];
        this.actionsGroups = [];
    }
}
function cleanupClipboardData(data) {
    return data.replace(/&nbsp;/g, "").replace(/(?:\r\n|\r|\n)/g, "<br />");
}
export async function pasteFromClipboard(position, elementRef) {
    const clipboardContent = await navigator.clipboard.readText();
    if (clipboardContent.startsWith("fromEden:")) {
        const content = clipboardContent.replace("fromEden:", "");
        elementRef.insertAdjacentHTML(position, content);
    }
    const newElement = {
        beforebegin: elementRef.previousElementSibling,
        afterbegin: elementRef.firstChild,
        afterend: elementRef.nextElementSibling,
    };
    smoothScrollTo(newElement[position]);
}
export function clipboardError() {
    document.body.appendChild(new EdenModal({
        title: t("snackbars.config-error-title"),
        content: t("snackbars.clipboard-config-error"),
    }));
}
async function recursInit(block, node, isCreation, parentSelector = "", parentBricks = []) {
    let lst = {};
    await Promise.all(Object.entries(node).map(async ([templateName, template]) => {
        let selector = parentSelector + templateName;
        let selectorIndex = lst[selector] || 0;
        lst[selector] = selectorIndex + 1;
        if (selectorIndex > 0) {
            selector += ":nth-child(" + selectorIndex + ")";
        }
        const element = block.element.querySelector(selector);
        let parents = parentBricks;
        const brickType = template.type;
        if (brickType !== undefined) {
            const { options, markup } = template;
            const Template = Templates[brickType];
            const tableParams = brickType === "Table" ? { rows: markup } : {};
            const recursiveParams = brickType === "Recursive" ? { templates: markup } : {};
            let brick = new Template({
                block,
                element,
                parentBricks: parents,
                options,
                markup,
                isCreation,
                ...tableParams,
                ...recursiveParams,
            });
            if (brickType === "Recursive" && typeof brick.init === "function") {
                await brick.init();
            }
            block.instances = [...(block.instances || []), brick];
            parents = [...parents, brick];
        }
        const markupType = typeOf(template.markup);
        if ((markupType === "undefined" && brickType === undefined) ||
            markupType === "object") {
            let next = markupType === "object" ? template.markup : template;
            await recursInit(block, next, isCreation, parentSelector + templateName + " > ", parents);
        }
    }));
}
function toggleDetails(e) {
    e.preventDefault();
    const target = e.target;
    const details = target.closest("details");
    if (!details)
        return;
    const open = details.hasAttribute("open");
    if (open) {
        details.removeAttribute("open");
    }
    else {
        details.setAttribute("open", "");
    }
}
function patchDetails(element) {
    const details = Array.from(element.querySelectorAll("a details"));
    details.forEach((detail) => detail.addEventListener("click", toggleDetails));
}
function cleanupDetailsPatch(element) {
    const details = Array.from(element.querySelectorAll("a details"));
    details.forEach((detail) => detail.removeEventListener("click", toggleDetails));
}
function setViewTransitionStyles() {
    const blocks = Array.from(document.querySelectorAll("section[data-template]"));
    blocks.forEach((element, i) => {
        element.style.setProperty("view-transition-name", `t-${i.toString().padStart(5, "0")}`);
    });
}
function removeViewTransitionStyles() {
    const delay = getViewTransitionDuration();
    setTimeout(() => {
        const blocks = Array.from(document.querySelectorAll("section[data-template]"));
        blocks.forEach((element) => {
            element.style.removeProperty("view-transition-name");
        });
    }, delay);
}
function getViewTransitionDuration() {
    const bodyStyles = window.getComputedStyle(document.body);
    const viewTransitionDuration = bodyStyles.getPropertyValue("--eden-view-transition-duration");
    return parseInt(viewTransitionDuration, 10);
}
function formateDate(date) {
    return new Date(date).toLocaleDateString("fr-FR", {
        year: "2-digit",
        month: "2-digit",
        day: "2-digit",
        hour: "2-digit",
        minute: "2-digit",
    });
}
