import { describe, it, expect, vi, beforeEach } from "vitest";
import Brick, { setEditable, unsetEditable } from "../brick";
describe("Brick", () => {
    let mockElement;
    let mockBlock;
    beforeEach(() => {
        mockElement = document.createElement("div");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
    });
    describe("constructor", () => {
        it("should initialize brick with element and block", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            expect(brick.element).toBe(mockElement);
            expect(brick.block).toBe(mockBlock);
            expect(brick.descriptor).toBe("");
        });
        it("should mark block as obsolete if element is null", () => {
            new Brick({
                block: mockBlock,
                element: null,
                parentBricks: [],
            });
            expect(mockBlock.obsolete).toBe(true);
        });
        it("should initialize with parent bricks", () => {
            const parentBrick = new Brick({
                block: mockBlock,
                element: document.createElement("div"),
                parentBricks: [],
            });
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [parentBrick],
            });
            expect(brick.parentBricks).toEqual([parentBrick]);
        });
    });
    describe("cleanRefs", () => {
        it("should clean block and parentBricks references", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            brick.cleanRefs();
            expect(brick.block).toBeUndefined();
            expect(brick.parentBricks).toBeUndefined();
        });
    });
    describe("handleClick", () => {
        it("should prevent default and stop propagation", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            const mockEvent = {
                preventDefault: vi.fn(),
                stopPropagation: vi.fn(),
            };
            brick.handleClick(mockEvent);
            expect(mockEvent.preventDefault).toHaveBeenCalled();
            expect(mockEvent.stopPropagation).toHaveBeenCalled();
        });
        it("should call block.manageActions with brick chain", () => {
            const parentBrick = new Brick({
                block: mockBlock,
                element: document.createElement("div"),
                parentBricks: [],
            });
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [parentBrick],
            });
            brick.handleClick();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([
                parentBrick,
                brick,
            ]);
        });
        it("should work without event parameter", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            expect(() => brick.handleClick()).not.toThrow();
        });
    });
    describe("hasParents", () => {
        it("should find parent element with matching nodeName", () => {
            const divParent = document.createElement("div");
            const spanChild = document.createElement("span");
            divParent.appendChild(spanChild);
            const brick = new Brick({
                block: mockBlock,
                element: divParent,
                parentBricks: [],
            });
            const result = brick.hasParents(spanChild, "div");
            expect(result).toBe(divParent);
        });
        it("should return null if target is null", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            const result = brick.hasParents(null, "div");
            expect(result).toBeNull();
        });
        it("should return target if nodeName matches", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            const div = document.createElement("div");
            const result = brick.hasParents(div, "div");
            expect(result).toBe(div);
        });
        it("should return undefined if reached element without finding match", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            const result = brick.hasParents(mockElement, "nonexistent");
            expect(result).toBeUndefined();
        });
    });
    describe("lifecycle methods", () => {
        it("should have preSave method", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            expect(typeof brick.preSave).toBe("function");
            expect(() => brick.preSave()).not.toThrow();
        });
        it("should have postSave method", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            expect(typeof brick.postSave).toBe("function");
            expect(() => brick.postSave()).not.toThrow();
        });
        it("should have activate method", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            expect(typeof brick.activate).toBe("function");
            expect(() => brick.activate()).not.toThrow();
        });
        it("should have desactivate method", () => {
            const brick = new Brick({
                block: mockBlock,
                element: mockElement,
                parentBricks: [],
            });
            expect(typeof brick.desactivate).toBe("function");
            expect(() => brick.desactivate()).not.toThrow();
        });
    });
});
describe("setEditable", () => {
    it("should set contentEditable to true on a single element", () => {
        const element = document.createElement("div");
        setEditable(element);
        expect(element.getAttribute("contentEditable")).toBe("true");
    });
    it("should set contentEditable to true on multiple elements", () => {
        const elements = [
            document.createElement("div"),
            document.createElement("span"),
            document.createElement("p"),
        ];
        setEditable(elements);
        elements.forEach((el) => {
            expect(el.getAttribute("contentEditable")).toBe("true");
        });
    });
    it("should handle undefined gracefully", () => {
        expect(() => setEditable(undefined)).not.toThrow();
    });
    it("should handle empty array", () => {
        expect(() => setEditable([])).not.toThrow();
    });
});
describe("unsetEditable", () => {
    it("should remove contentEditable attribute from a single element", () => {
        const element = document.createElement("div");
        element.setAttribute("contentEditable", "true");
        unsetEditable(element);
        expect(element.hasAttribute("contentEditable")).toBe(false);
    });
    it("should remove contentEditable from multiple elements", () => {
        const elements = [
            document.createElement("div"),
            document.createElement("span"),
            document.createElement("p"),
        ];
        elements.forEach((el) => el.setAttribute("contentEditable", "true"));
        unsetEditable(elements);
        elements.forEach((el) => {
            expect(el.hasAttribute("contentEditable")).toBe(false);
        });
    });
    it("should handle undefined gracefully", () => {
        expect(() => unsetEditable(undefined)).not.toThrow();
    });
    it("should handle empty array", () => {
        expect(() => unsetEditable([])).not.toThrow();
    });
    it("should not throw if attribute doesn't exist", () => {
        const element = document.createElement("div");
        expect(() => unsetEditable(element)).not.toThrow();
    });
});
