import { describe, it, expect, beforeEach, vi } from "vitest";
import Text from "../Text";
vi.mock("../../user-interface/block/eden-text-helper", () => ({
    default: class MockEdenTextHelper {
        constructor() {
            this.init = vi.fn();
            this.update = vi.fn();
            this.remove = vi.fn();
            this.setAttribute = vi.fn();
            this.removeAttribute = vi.fn();
        }
    },
}));
describe("Text Brick", () => {
    let mockBlock;
    let element;
    beforeEach(() => {
        element = document.createElement("div");
        element.contentEditable = "true";
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        window.EdenConfig = {
            allowClearTextEmail: false,
        };
    });
    describe("constructor", () => {
        it("should initialize Text brick", () => {
            const text = new Text({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(text.element).toBe(element);
            expect(text.block).toBe(mockBlock);
            expect(text.actions).toEqual([]);
        });
        it("should initialize with textLength options", () => {
            const options = {
                textLength: {
                    max: 100,
                },
            };
            const text = new Text({
                block: mockBlock,
                element,
                options,
                parentBricks: [],
            });
            expect(text.options).toEqual(options);
            expect(text.hasTextLengthOption).toBe(100);
        });
        it("should handle missing options", () => {
            const text = new Text({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(text.options).toBeUndefined();
            expect(text.hasTextLengthOption).toBeUndefined();
        });
    });
    describe("updateTextLengthHelper", () => {
        it("should update helper content", () => {
            const options = {
                textLength: { max: 50 },
            };
            const text = new Text({
                block: mockBlock,
                element,
                options,
                parentBricks: [],
            });
            element.textContent = "Test content";
            text.textLengthHelper = {
                update: vi.fn(),
                setAttribute: vi.fn(),
                removeAttribute: vi.fn(),
            };
            const event = new Event("input");
            text.updateTextLengthHelper(event);
            expect(text.textLengthHelper?.update).toHaveBeenCalled();
        });
    });
    describe("removeTextLengthHelper", () => {
        it("should remove helper if it exists", () => {
            const text = new Text({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            text.textLengthHelper = {
                remove: vi.fn(),
            };
            text.removeTextLengthHelper();
            expect(text.textLengthHelper?.remove).toHaveBeenCalled();
        });
        it("should not throw if helper doesn't exist", () => {
            const text = new Text({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => text.removeTextLengthHelper()).not.toThrow();
        });
    });
    describe("removeEmailPresenceHelper", () => {
        it("should remove email helper and set to undefined", () => {
            const text = new Text({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            text.emailPresenceHelper = {
                remove: vi.fn(),
            };
            text.removeEmailPresenceHelper();
            expect(text.emailPresenceHelper).toBeUndefined();
        });
        it("should not throw if helper doesn't exist", () => {
            const text = new Text({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => text.removeEmailPresenceHelper()).not.toThrow();
        });
    });
});
