import { describe, it, expect, beforeEach, vi, afterEach } from "vitest";
import Table from "../Table";
vi.mock("../../services/html", () => ({
    htmlStringToElement: vi.fn((html) => {
        const template = document.createElement("template");
        template.innerHTML = html.trim();
        return template.content.firstChild;
    }),
}));
vi.mock("../../user-interface/common/eden-snackbar", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-modal", () => ({
    default: vi.fn().mockImplementation(function (config) {
        this.config = config;
        const element = document.createElement("div");
        element.className = "eden-modal";
        Object.assign(element, this);
        return element;
    }),
}));
vi.mock("../../i18n/index", () => ({
    default: vi.fn((key) => key),
}));
import { htmlStringToElement } from "../../services/html";
import { snackbar } from "../../user-interface/common/eden-snackbar";
import EdenModal from "../../user-interface/common/eden-modal";
describe("Table Brick", () => {
    let mockBlock;
    let element;
    let rows;
    beforeEach(() => {
        element = document.createElement("table");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        rows = [
            {
                name: "Row Type 1",
                markup: "<tr data-rowformat='0'><td>Cell 1</td><td>Cell 2</td></tr>",
            },
            {
                name: "Row Type 2",
                markup: "<tr data-rowformat='1'><td>Header 1</td><td>Header 2</td></tr>",
            },
        ];
        vi.clearAllMocks();
        document.body.innerHTML = "";
    });
    afterEach(() => {
        document.body.innerHTML = "";
    });
    describe("constructor", () => {
        it("should initialize Table brick with rows", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            expect(table.element).toBe(element);
            expect(table.block).toBe(mockBlock);
            expect(table.rows).toBe(rows);
            expect(table.tools).toBe("");
            expect(table.actions).toHaveLength(5);
        });
        it("should throw error if no rows specified", () => {
            expect(() => {
                new Table({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                    rows: undefined,
                });
            }).toThrow("No row specified");
        });
        it("should create actions with correct configuration", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            expect(table.actions[0].icon).toBe("plus");
            expect(table.actions[0].tooltip).toBe("bricks.table.add-row");
            expect(table.actions[1].icon).toBe("duplicate");
            expect(table.actions[2].icon).toBe("chevron_up");
            expect(table.actions[3].icon).toBe("chevron_down");
            expect(table.actions[4].icon).toBe("delete");
        });
        it("should create add-row action with dropdown for each row type", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const addRowAction = table.actions[0];
            expect(Array.isArray(addRowAction.action)).toBe(true);
            const actionArray = addRowAction.action;
            expect(actionArray).toHaveLength(2);
            expect(actionArray[0].name).toBe("Row Type 1");
            expect(actionArray[1].name).toBe("Row Type 2");
        });
    });
    describe("selectedLine", () => {
        it("should return the selected tr element", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "test";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const result = table.selectedLine();
            expect(result).toBe(tr);
        });
        it("should return null if no selection", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            window.getSelection()?.removeAllRanges();
            const result = table.selectedLine();
            expect(result).toBeNull();
        });
        it("should return null if selection is not in a tr", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const div = document.createElement("div");
            div.textContent = "test";
            document.body.appendChild(div);
            const range = document.createRange();
            range.selectNodeContents(div);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const result = table.selectedLine();
            expect(result).toBeNull();
        });
    });
    describe("applyOnLine", () => {
        it("should apply function to selected line", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "Test";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const mockFn = vi.fn();
            table.applyOnLine(mockFn);
            expect(mockFn).toHaveBeenCalledWith(tr);
        });
        it("should show snackbar warning if no line is selected", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            window.getSelection()?.removeAllRanges();
            const mockFn = vi.fn();
            table.applyOnLine(mockFn);
            expect(snackbar).toHaveBeenCalledWith("snackbars.select-line", "warning");
            expect(mockFn).not.toHaveBeenCalled();
        });
        it("should show snackbar when applyOnLine is called with null selection", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.selectedLine = vi.fn().mockReturnValue(null);
            const mockFn = vi.fn();
            table.applyOnLine(mockFn);
            expect(mockFn).not.toHaveBeenCalled();
            expect(snackbar).toHaveBeenCalledWith("snackbars.select-line", "warning");
        });
        it("should focus first child after applying function", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "Test";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const focusSpy = vi.spyOn(td, "focus");
            table.applyOnLine(() => { });
            expect(focusSpy).toHaveBeenCalled();
        });
        it("should not throw if first child does not exist", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const textNode = document.createTextNode("text");
            tr.appendChild(textNode);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(textNode);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            expect(() => {
                table.applyOnLine(() => { });
            }).not.toThrow();
        });
    });
    describe("add row action", () => {
        it("should add a new row after the selected row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const existingTr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "test";
            existingTr.appendChild(td);
            element.appendChild(existingTr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const addRowActions = table.actions[0].action;
            addRowActions[0].action();
            expect(element.querySelectorAll("tr")).toHaveLength(2);
        });
        it("should add row at the end if no row is selected", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.selectedLine = vi.fn().mockReturnValue(null);
            const addRowActions = table.actions[0].action;
            addRowActions[0].action();
            expect(element.querySelectorAll("tr")).toHaveLength(1);
        });
        it("should set correct rowformat data attribute", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.selectedLine = vi.fn().mockReturnValue(null);
            const addRowActions = table.actions[0].action;
            addRowActions[1].action();
            const newRow = element.querySelector("tr");
            expect(newRow?.dataset.rowformat).toBe("1");
        });
        it("should make cells editable after adding row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.selectedLine = vi.fn().mockReturnValue(null);
            const addRowActions = table.actions[0].action;
            addRowActions[0].action();
            const cells = element.querySelectorAll("td:not(.link), th:not(.link)");
            cells.forEach((cell) => {
                expect(cell.getAttribute("contentEditable")).toBe("true");
            });
        });
        it("should focus first child after adding row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            document.body.appendChild(element);
            table.selectedLine = vi.fn().mockReturnValue(null);
            const addRowActions = table.actions[0].action;
            addRowActions[0].action();
            const newRow = element.querySelector("tr");
            const firstChild = newRow?.querySelector(":first-child");
            expect(document.activeElement).toBe(firstChild);
        });
        it("should not add row if markup is invalid", () => {
            vi.mocked(htmlStringToElement).mockReturnValueOnce(null);
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const addRowActions = table.actions[0].action;
            addRowActions[0].action();
            expect(element.querySelectorAll("tr")).toHaveLength(0);
        });
    });
    describe("duplicate action", () => {
        it("should duplicate the selected row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "Test Content";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const duplicateAction = table.actions[1].action;
            duplicateAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows).toHaveLength(2);
            expect(tableRows[0].textContent).toBe(tableRows[1].textContent);
        });
        it("should insert duplicate after the original row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr1 = document.createElement("tr");
            const td1 = document.createElement("td");
            td1.textContent = "First";
            tr1.appendChild(td1);
            const tr2 = document.createElement("tr");
            const td2 = document.createElement("td");
            td2.textContent = "Second";
            tr2.appendChild(td2);
            element.appendChild(tr1);
            element.appendChild(tr2);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td1);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const duplicateAction = table.actions[1].action;
            duplicateAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows).toHaveLength(3);
            expect(tableRows[0].textContent).toBe("First");
            expect(tableRows[1].textContent).toBe("First");
            expect(tableRows[2].textContent).toBe("Second");
        });
    });
    describe("move up action", () => {
        it("should move selected row up", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr1 = document.createElement("tr");
            const td1 = document.createElement("td");
            td1.textContent = "First";
            tr1.appendChild(td1);
            const tr2 = document.createElement("tr");
            const td2 = document.createElement("td");
            td2.textContent = "Second";
            tr2.appendChild(td2);
            element.appendChild(tr1);
            element.appendChild(tr2);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td2);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const moveUpAction = table.actions[2].action;
            moveUpAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows[0].textContent).toBe("Second");
            expect(tableRows[1].textContent).toBe("First");
        });
        it("should not move row if it's already at the top", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "First";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const moveUpAction = table.actions[2].action;
            moveUpAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows).toHaveLength(1);
            expect(tableRows[0].textContent).toBe("First");
        });
    });
    describe("move down action", () => {
        it("should move selected row down", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr1 = document.createElement("tr");
            const td1 = document.createElement("td");
            td1.textContent = "First";
            tr1.appendChild(td1);
            const tr2 = document.createElement("tr");
            const td2 = document.createElement("td");
            td2.textContent = "Second";
            tr2.appendChild(td2);
            element.appendChild(tr1);
            element.appendChild(tr2);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td1);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const moveDownAction = table.actions[3].action;
            moveDownAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows[0].textContent).toBe("Second");
            expect(tableRows[1].textContent).toBe("First");
        });
        it("should not move row if it's already at the bottom", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.textContent = "Only";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const moveDownAction = table.actions[3].action;
            moveDownAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows).toHaveLength(1);
            expect(tableRows[0].textContent).toBe("Only");
        });
    });
    describe("remove action", () => {
        it("should show confirmation modal when removing row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr1 = document.createElement("tr");
            const td1 = document.createElement("td");
            tr1.appendChild(td1);
            const tr2 = document.createElement("tr");
            const td2 = document.createElement("td");
            tr2.appendChild(td2);
            element.appendChild(tr1);
            element.appendChild(tr2);
            const removeAction = table.actions[4].action;
            removeAction();
            expect(EdenModal).toHaveBeenCalled();
            const modalInstance = vi.mocked(EdenModal).mock.results[0].value;
            expect(modalInstance.config.icon).toBe("delete");
            expect(modalInstance.config.title).toBe("bricks.table.remove-row");
        });
        it("should remove row when confirmed", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr1 = document.createElement("tr");
            const td1 = document.createElement("td");
            td1.textContent = "First";
            tr1.appendChild(td1);
            const tr2 = document.createElement("tr");
            const td2 = document.createElement("td");
            td2.textContent = "Second";
            tr2.appendChild(td2);
            element.appendChild(tr1);
            element.appendChild(tr2);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td1);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const removeAction = table.actions[4].action;
            removeAction();
            const modalInstance = vi.mocked(EdenModal).mock.results[0].value;
            const confirmButton = modalInstance.config.buttons[1];
            confirmButton.onClick();
            expect(element.querySelectorAll("tr")).toHaveLength(1);
            expect(element.querySelector("tr")?.textContent).toBe("Second");
        });
        it("should prevent removing last row", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            tr.appendChild(td);
            element.appendChild(tr);
            const removeAction = table.actions[4].action;
            removeAction();
            expect(snackbar).toHaveBeenCalledWith("snackbars.cannot-delete-last-row", "warning");
            expect(EdenModal).not.toHaveBeenCalled();
        });
        it("should have correct modal buttons configuration", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr1 = document.createElement("tr");
            const tr2 = document.createElement("tr");
            element.appendChild(tr1);
            element.appendChild(tr2);
            const removeAction = table.actions[4].action;
            removeAction();
            const modalInstance = vi.mocked(EdenModal).mock.results[0].value;
            expect(modalInstance.config.buttons).toHaveLength(2);
            expect(modalInstance.config.buttons[0].value).toBe("common.cancel");
            expect(modalInstance.config.buttons[0].cssClass).toBe("cancel");
            expect(modalInstance.config.buttons[1].value).toBe("common.remove");
            expect(modalInstance.config.buttons[1].cssClass).toBe("confirm");
            expect(modalInstance.config.buttons[1].autofocus).toBe(true);
            expect(modalInstance.config.closeButton).toBe(false);
        });
    });
    describe("activate", () => {
        it("should add click event listener and make cells editable", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const td1 = document.createElement("td");
            const td2 = document.createElement("td");
            const th1 = document.createElement("th");
            element.appendChild(td1);
            element.appendChild(td2);
            element.appendChild(th1);
            table.activate();
            expect(td1.getAttribute("contentEditable")).toBe("true");
            expect(td2.getAttribute("contentEditable")).toBe("true");
            expect(th1.getAttribute("contentEditable")).toBe("true");
        });
        it("should not make link cells editable", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const td = document.createElement("td");
            td.classList.add("link");
            element.appendChild(td);
            table.activate();
            expect(td.getAttribute("contentEditable")).toBeNull();
        });
        it("should handle click events", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const handleClickSpy = vi.spyOn(table, "handleClick");
            table.activate();
            const event = new MouseEvent("click");
            element.dispatchEvent(event);
            expect(handleClickSpy).toHaveBeenCalled();
        });
        it("should not throw if element is null", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.element = null;
            expect(() => table.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener and unset editable cells", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const td1 = document.createElement("td");
            const td2 = document.createElement("td");
            td1.setAttribute("contentEditable", "true");
            td2.setAttribute("contentEditable", "true");
            element.appendChild(td1);
            element.appendChild(td2);
            table.desactivate();
            expect(td1.hasAttribute("contentEditable")).toBe(false);
            expect(td2.hasAttribute("contentEditable")).toBe(false);
        });
        it("should not affect link cells", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const td = document.createElement("td");
            td.classList.add("link");
            td.setAttribute("contentEditable", "true");
            element.appendChild(td);
            table.desactivate();
            expect(td.hasAttribute("contentEditable")).toBe(true);
        });
        it("should remove event listener properly", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.activate();
            const handleClickSpy = vi.spyOn(table, "handleClick");
            table.desactivate();
            const event = new MouseEvent("click");
            element.dispatchEvent(event);
            expect(handleClickSpy).not.toHaveBeenCalled();
        });
        it("should not throw if element is null", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.element = null;
            expect(() => table.desactivate()).not.toThrow();
        });
    });
    describe("integration tests", () => {
        it("should handle complete workflow: activate, add row, duplicate, move, remove", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            document.body.appendChild(element);
            table.activate();
            const selectedLineSpy = vi.spyOn(table, "selectedLine");
            selectedLineSpy.mockReturnValueOnce(null);
            const addRowActions = table.actions[0].action;
            addRowActions[0].action();
            expect(element.querySelectorAll("tr")).toHaveLength(1);
            const firstRow = element.querySelector("tr");
            const firstCell = firstRow?.querySelector("td");
            if (firstCell) {
                firstCell.textContent = "test";
                const range = document.createRange();
                range.selectNodeContents(firstCell);
                const selection = window.getSelection();
                selection?.removeAllRanges();
                selection?.addRange(range);
            }
            addRowActions[0].action();
            expect(element.querySelectorAll("tr")).toHaveLength(2);
            const duplicateAction = table.actions[1].action;
            duplicateAction();
            expect(element.querySelectorAll("tr")).toHaveLength(3);
            const moveDownAction = table.actions[3].action;
            moveDownAction();
            expect(element.querySelectorAll("tr")).toHaveLength(3);
            const rowToDelete = element.querySelectorAll("tr")[1];
            const cellToSelect = rowToDelete.querySelector("td");
            if (cellToSelect) {
                const range = document.createRange();
                range.selectNodeContents(cellToSelect);
                const selection = window.getSelection();
                selection?.removeAllRanges();
                selection?.addRange(range);
            }
            vi.clearAllMocks();
            const removeAction = table.actions[4].action;
            removeAction();
            const modalInstance = vi.mocked(EdenModal).mock.results[0].value;
            modalInstance.config.buttons[1].onClick();
            expect(element.querySelectorAll("tr")).toHaveLength(2);
            table.desactivate();
            const cells = element.querySelectorAll("td");
            cells.forEach((cell) => {
                expect(cell.hasAttribute("contentEditable")).toBe(false);
            });
        });
        it("should properly inherit from Brick class", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            expect(table.element).toBe(element);
            expect(table.block).toBe(mockBlock);
            expect(typeof table.handleClick).toBe("function");
            expect(typeof table.cleanRefs).toBe("function");
            expect(typeof table.preSave).toBe("function");
            expect(typeof table.postSave).toBe("function");
        });
    });
    describe("edge cases", () => {
        it("should handle multiple rapid row additions", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.selectedLine = vi.fn().mockReturnValue(null);
            const addRowActions = table.actions[0].action;
            for (let i = 0; i < 10; i++) {
                addRowActions[0].action();
            }
            expect(element.querySelectorAll("tr")).toHaveLength(10);
        });
        it("should handle row operations with empty table", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            table.selectedLine = vi.fn().mockImplementation(() => {
                throw new DOMException("Invalid range index.", "IndexSizeError");
            });
            const duplicateAction = table.actions[1].action;
            expect(() => duplicateAction()).toThrow();
        });
        it("should handle cells with complex content", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            const td = document.createElement("td");
            td.innerHTML = "<strong>Bold</strong> <em>Italic</em>";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const duplicateAction = table.actions[1].action;
            duplicateAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows).toHaveLength(2);
            expect(tableRows[1].innerHTML).toContain("<strong>Bold</strong>");
            expect(tableRows[1].innerHTML).toContain("<em>Italic</em>");
        });
        it("should maintain data attributes when duplicating rows", () => {
            const table = new Table({
                block: mockBlock,
                element,
                parentBricks: [],
                rows,
            });
            const tr = document.createElement("tr");
            tr.dataset.customAttribute = "test-value";
            tr.dataset.rowformat = "2";
            const td = document.createElement("td");
            td.textContent = "test";
            tr.appendChild(td);
            element.appendChild(tr);
            document.body.appendChild(element);
            const range = document.createRange();
            range.selectNodeContents(td);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const duplicateAction = table.actions[1].action;
            duplicateAction();
            const tableRows = element.querySelectorAll("tr");
            expect(tableRows[1].dataset.customAttribute).toBe("test-value");
            expect(tableRows[1].dataset.rowformat).toBe("2");
        });
    });
});
