import { describe, it, expect, beforeEach, vi, afterEach } from "vitest";
import Spotify from "../Spotify";
vi.mock("../../user-interface/plugin/spotify-modal.js", () => ({
    default: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-snackbar.js", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../i18n/index.js", () => ({
    default: vi.fn((key) => key),
}));
import SpotifyModal from "../../user-interface/plugin/spotify-modal.js";
import { snackbar } from "../../user-interface/common/eden-snackbar.js";
describe("Spotify Brick", () => {
    let mockBlock;
    let mockElement;
    let mockParentBricks;
    let constructorParams;
    beforeEach(() => {
        vi.clearAllMocks();
        mockBlock = {
            element: document.createElement("div"),
            obsolete: false,
            manageActions: vi.fn(),
        };
        mockElement = document.createElement("div");
        mockParentBricks = [];
        constructorParams = {
            block: mockBlock,
            element: mockElement,
            parentBricks: mockParentBricks,
        };
    });
    afterEach(() => {
        document.body.innerHTML = "";
    });
    describe("Constructor", () => {
        it("should initialize with correct properties", () => {
            const spotify = new Spotify(constructorParams);
            expect(spotify.element).toBe(mockElement);
            expect(spotify.block).toBe(mockBlock);
            expect(spotify.parentBricks).toBe(mockParentBricks);
            expect(spotify.mask).toBeInstanceOf(HTMLDivElement);
        });
        it("should create mask element with correct styles", () => {
            const spotify = new Spotify(constructorParams);
            expect(spotify.mask.style.position).toBe("absolute");
            expect(spotify.mask.style.top).toBe("0px");
            expect(spotify.mask.style.bottom).toBe("0px");
            expect(spotify.mask.style.left).toBe("0px");
            expect(spotify.mask.style.right).toBe("0px");
        });
        it("should initialize with actions array", () => {
            const spotify = new Spotify(constructorParams);
            expect(spotify.actions).toBeDefined();
            expect(spotify.actions).toHaveLength(1);
            expect(spotify.actions[0]).toHaveProperty("tooltip");
            expect(spotify.actions[0]).toHaveProperty("icon", "playlist");
            expect(spotify.actions[0]).toHaveProperty("action");
            expect(typeof spotify.actions[0].action).toBe("function");
        });
    });
    describe("Actions", () => {
        it("should open SpotifyModal when action is triggered without existing iframe", () => {
            const spotify = new Spotify(constructorParams);
            const actionFn = spotify.actions[0].action;
            actionFn();
            expect(SpotifyModal).toHaveBeenCalledWith(spotify.updateUrl, "", "300", "380");
        });
        it("should open SpotifyModal with existing iframe data", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://open.spotify.com/embed/playlist/abc123");
            iframe.setAttribute("width", "400");
            iframe.setAttribute("height", "500");
            mockElement.appendChild(iframe);
            const actionFn = spotify.actions[0].action;
            actionFn();
            expect(SpotifyModal).toHaveBeenCalledWith(spotify.updateUrl, "spotify:playlist:abc123", "400", "500");
        });
        it("should use default values if iframe has no attributes", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            const actionFn = spotify.actions[0].action;
            actionFn();
            expect(SpotifyModal).toHaveBeenCalledWith(spotify.updateUrl, "", "300", "380");
        });
    });
    describe("updateUrl", () => {
        it("should update existing iframe with valid uri", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.updateUrl("spotify:track:123abc", "350", "450");
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/track/123abc");
            expect(iframe.getAttribute("width")).toBe("350");
            expect(iframe.getAttribute("height")).toBe("450");
            expect(iframe.style.maxWidth).toBe("100%");
            expect(iframe.getAttribute("frameborder")).toBe("0");
            expect(iframe.getAttribute("allowtransparency")).toBe("true");
            expect(iframe.getAttribute("allow")).toBe("encrypted-media");
        });
        it("should create new iframe if none exists", () => {
            const spotify = new Spotify(constructorParams);
            spotify.updateUrl("spotify:album:xyz789", "300", "380");
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/album/xyz789");
            expect(iframe.getAttribute("width")).toBe("300");
            expect(iframe.getAttribute("height")).toBe("380");
            expect(iframe.style.maxWidth).toBe("100%");
            const maskInElement = mockElement.contains(spotify.mask);
            expect(maskInElement).toBe(true);
        });
        it("should handle playlist uri correctly", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.updateUrl("spotify:playlist:37i9dQZF1DXcBWIGoYBM5M", "400", "500");
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/playlist/37i9dQZF1DXcBWIGoYBM5M");
        });
        it("should show error snackbar for invalid uri (missing type)", () => {
            const spotify = new Spotify(constructorParams);
            spotify.updateUrl("spotify:", "300", "380");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-uri", "error");
        });
        it("should show error snackbar for invalid uri (missing code)", () => {
            const spotify = new Spotify(constructorParams);
            spotify.updateUrl("spotify:track", "300", "380");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-uri", "error");
        });
        it("should show error snackbar for completely invalid uri", () => {
            const spotify = new Spotify(constructorParams);
            spotify.updateUrl("invalid-uri", "300", "380");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-uri", "error");
        });
        it("should show error snackbar for empty uri", () => {
            const spotify = new Spotify(constructorParams);
            spotify.updateUrl("", "300", "380");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-uri", "error");
        });
        it("should handle episode uri correctly", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.updateUrl("spotify:episode:abc123def456", "300", "232");
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/episode/abc123def456");
            expect(iframe.getAttribute("width")).toBe("300");
            expect(iframe.getAttribute("height")).toBe("232");
        });
        it("should handle show uri correctly", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.updateUrl("spotify:show:showid123", "300", "232");
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/show/showid123");
        });
    });
    describe("activate", () => {
        it("should add click event listener to element", () => {
            const spotify = new Spotify(constructorParams);
            const addEventListenerSpy = vi.spyOn(mockElement, "addEventListener");
            spotify.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", spotify.handleClick);
        });
        it("should append mask if iframe exists", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.activate();
            expect(mockElement.contains(spotify.mask)).toBe(true);
        });
        it("should not append mask if no iframe exists", () => {
            const spotify = new Spotify(constructorParams);
            spotify.activate();
            expect(mockElement.contains(spotify.mask)).toBe(false);
        });
        it("should do nothing if element is null", () => {
            const nullElementParams = {
                ...constructorParams,
                element: null,
            };
            const spotify = new Spotify(nullElementParams);
            expect(() => spotify.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener from element", () => {
            const spotify = new Spotify(constructorParams);
            const removeEventListenerSpy = vi.spyOn(mockElement, "removeEventListener");
            spotify.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", spotify.handleClick);
        });
        it("should remove mask from DOM", () => {
            const spotify = new Spotify(constructorParams);
            mockElement.appendChild(spotify.mask);
            spotify.desactivate();
            expect(mockElement.contains(spotify.mask)).toBe(false);
        });
        it("should not throw if mask is not in DOM", () => {
            const spotify = new Spotify(constructorParams);
            expect(() => spotify.desactivate()).not.toThrow();
        });
        it("should do nothing if element is null", () => {
            const nullElementParams = {
                ...constructorParams,
                element: null,
            };
            const spotify = new Spotify(nullElementParams);
            expect(() => spotify.desactivate()).not.toThrow();
        });
    });
    describe("Integration: activate and desactivate cycle", () => {
        it("should properly add and remove mask during activate/desactivate cycle", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.activate();
            expect(mockElement.contains(spotify.mask)).toBe(true);
            spotify.desactivate();
            expect(mockElement.contains(spotify.mask)).toBe(false);
            spotify.activate();
            expect(mockElement.contains(spotify.mask)).toBe(true);
        });
    });
    describe("handleClick", () => {
        it("should call manageActions when clicked", () => {
            const spotify = new Spotify(constructorParams);
            const mockEvent = new MouseEvent("click");
            const preventDefaultSpy = vi.spyOn(mockEvent, "preventDefault");
            const stopPropagationSpy = vi.spyOn(mockEvent, "stopPropagation");
            spotify.handleClick(mockEvent);
            expect(preventDefaultSpy).toHaveBeenCalled();
            expect(stopPropagationSpy).toHaveBeenCalled();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([spotify]);
        });
        it("should include parent bricks in manageActions call", () => {
            const parentBrick = { element: document.createElement("div") };
            const paramsWithParent = {
                ...constructorParams,
                parentBricks: [parentBrick],
            };
            const spotify = new Spotify(paramsWithParent);
            const mockEvent = new MouseEvent("click");
            spotify.handleClick(mockEvent);
            expect(mockBlock.manageActions).toHaveBeenCalledWith([
                parentBrick,
                spotify,
            ]);
        });
        it("should work without event parameter", () => {
            const spotify = new Spotify(constructorParams);
            expect(() => spotify.handleClick()).not.toThrow();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([spotify]);
        });
    });
    describe("Edge cases and error handling", () => {
        it("should handle updateUrl with iframe but invalid uri", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://open.spotify.com/embed/track/old123");
            mockElement.appendChild(iframe);
            spotify.updateUrl("not-a-valid-uri", "300", "380");
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/track/old123");
            expect(snackbar).toHaveBeenCalledWith("snackbars.fill-valid-uri", "error");
        });
        it("should handle multiple iframes in element", () => {
            const spotify = new Spotify(constructorParams);
            const iframe1 = document.createElement("iframe");
            const iframe2 = document.createElement("iframe");
            mockElement.appendChild(iframe1);
            mockElement.appendChild(iframe2);
            spotify.updateUrl("spotify:track:abc123", "300", "380");
            expect(iframe1.getAttribute("src")).toBe("https://open.spotify.com/embed/track/abc123");
            expect(iframe2.getAttribute("src")).toBeNull();
        });
        it("should properly replace element content when creating new iframe", () => {
            const spotify = new Spotify(constructorParams);
            const existingContent = document.createElement("div");
            existingContent.textContent = "Old content";
            mockElement.appendChild(existingContent);
            spotify.updateUrl("spotify:track:new123", "300", "380");
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(mockElement.querySelector("div:not([style*='position'])")).toBeNull();
        });
    });
    describe("Utility functions behavior", () => {
        it("should correctly parse different Spotify URI formats", () => {
            const spotify = new Spotify(constructorParams);
            const testCases = [
                {
                    uri: "spotify:track:3n3Ppam7vgaVa1iaRUc9Lp",
                    expectedSrc: "https://open.spotify.com/embed/track/3n3Ppam7vgaVa1iaRUc9Lp",
                },
                {
                    uri: "spotify:album:6QaVfG1pHYl1z15ZxkvVDW",
                    expectedSrc: "https://open.spotify.com/embed/album/6QaVfG1pHYl1z15ZxkvVDW",
                },
                {
                    uri: "spotify:playlist:37i9dQZF1DXcBWIGoYBM5M",
                    expectedSrc: "https://open.spotify.com/embed/playlist/37i9dQZF1DXcBWIGoYBM5M",
                },
                {
                    uri: "spotify:artist:0TnOYISbd1XYRBk9myaseg",
                    expectedSrc: "https://open.spotify.com/embed/artist/0TnOYISbd1XYRBk9myaseg",
                },
            ];
            testCases.forEach(({ uri, expectedSrc }) => {
                const iframe = document.createElement("iframe");
                mockElement.innerHTML = "";
                mockElement.appendChild(iframe);
                spotify.updateUrl(uri, "300", "380");
                expect(iframe.getAttribute("src")).toBe(expectedSrc);
            });
        });
        it("should handle buildUri reverse conversion", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            iframe.setAttribute("src", "https://open.spotify.com/embed/track/abc123");
            mockElement.appendChild(iframe);
            const actionFn = spotify.actions[0].action;
            actionFn();
            expect(SpotifyModal).toHaveBeenCalledWith(spotify.updateUrl, "spotify:track:abc123", "300", "380");
        });
    });
    describe("Attributes preservation", () => {
        it("should preserve all required iframe attributes on update", () => {
            const spotify = new Spotify(constructorParams);
            const iframe = document.createElement("iframe");
            mockElement.appendChild(iframe);
            spotify.updateUrl("spotify:track:test123", "400", "500");
            expect(iframe.getAttribute("frameborder")).toBe("0");
            expect(iframe.getAttribute("allowtransparency")).toBe("true");
            expect(iframe.getAttribute("allow")).toBe("encrypted-media");
            expect(iframe.style.maxWidth).toBe("100%");
        });
        it("should set all required attributes when creating new iframe", () => {
            const spotify = new Spotify(constructorParams);
            spotify.updateUrl("spotify:track:test123", "400", "500");
            const iframe = mockElement.querySelector("iframe");
            expect(iframe).not.toBeNull();
            expect(iframe.getAttribute("src")).toBe("https://open.spotify.com/embed/track/test123");
            expect(iframe.style.maxWidth).toBe("100%");
            expect(iframe.getAttribute("width")).toBe("400");
            expect(iframe.getAttribute("height")).toBe("500");
        });
    });
});
