import { describe, it, expect, vi, beforeEach, afterEach } from "vitest";
import RichTextFormat, { findParentFromSelection } from "../RichText.js";
vi.mock("../../core/brick.js", () => ({
    default: class Brick {
        constructor({ block, element, parentBricks }) {
            this.descriptor = "";
            this.actions = [];
            this.cleanRefs = vi.fn();
            this.handleClick = vi.fn();
            this.preSave = vi.fn();
            this.postSave = vi.fn();
            this.activate = vi.fn();
            this.desactivate = vi.fn();
            this.block = block;
            this.element = element;
            this.parentBricks = parentBricks;
        }
    },
    setEditable: vi.fn(),
    unsetEditable: vi.fn(),
}));
vi.mock("../../services/keys.js", () => ({
    generateShortcut: vi.fn((...keys) => keys.join("+")),
}));
vi.mock("../../services/html.js", () => ({
    default: vi.fn((strings, ...values) => {
        return strings.reduce((acc, str, i) => {
            const value = values[i] !== undefined ? String(values[i]) : "";
            return acc + str + value;
        }, "");
    }),
}));
vi.mock("../../i18n/index.js", () => ({
    default: vi.fn((key) => key),
}));
vi.mock("../Text.js", () => ({
    default: class TextFormat {
        constructor({ block, element, parentBricks }) {
            this.actions = [];
            this.handleClick = vi.fn();
            this.block = block;
            this.element = element;
            this.parentBricks = parentBricks;
        }
    },
}));
vi.mock("../../user-interface/icons/library/index.js", () => ({
    default: {},
}));
describe("RichTextFormat", () => {
    let element;
    let block;
    let richText;
    beforeEach(() => {
        document.body.innerHTML =
            '<div id="test-element" contenteditable="true"></div>';
        element = document.getElementById("test-element");
        Object.defineProperty(element, "innerText", {
            get() {
                return this.textContent || "";
            },
            configurable: true,
        });
        block = {
            element: document.createElement("div"),
        };
        window.EdenConfig = {
            styles: {},
        };
        Object.defineProperty(window, "location", {
            value: {
                protocol: "https:",
                host: "example.com",
            },
            writable: true,
            configurable: true,
        });
        document.execCommand = vi.fn(() => true);
    });
    afterEach(() => {
        vi.clearAllMocks();
        document.body.innerHTML = "";
        delete window.EdenConfig;
    });
    describe("Constructor", () => {
        it("should initialize with default actions when no options provided", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            expect(richText.element).toBe(element);
            expect(richText.watchSelectionChange).toBe(true);
            expect(richText.actions.length).toBeGreaterThan(0);
        });
        it("should filter actions based on options.commands", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    commands: ["bold", "italic"],
                },
            });
            expect(richText.actions.length).toBe(2);
            expect(richText.actions.every((a) => ["bold", "italic"].includes(a.code))).toBe(true);
        });
        it("should add source action when allowHtmlEdit is true", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    allowHtmlEdit: true,
                },
            });
            const sourceAction = richText.actions.find((a) => a.code === "source");
            expect(sourceAction).toBeDefined();
        });
        it("should set text length options", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    textLength: {
                        max: 100,
                        force: true,
                    },
                },
            });
            expect(richText.hasTextLengthOption).toBe(100);
            expect(richText.enforceTextLength).toBe(true);
        });
        it("should include color styles from config", () => {
            window.EdenConfig.styles = {
                color: [
                    { name: "Red", value: "#ff0000" },
                    { name: "Blue", value: "#0000ff" },
                ],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const colorAction = richText.actions.find((a) => a.code === "color");
            expect(colorAction).toBeDefined();
            expect(Array.isArray(colorAction?.action)).toBe(true);
        });
        it("should include background styles from config", () => {
            window.EdenConfig.styles = {
                background: [{ name: "Yellow", value: "#ffff00" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const bgAction = richText.actions.find((a) => a.code === "background");
            expect(bgAction).toBeDefined();
        });
        it("should include font family styles from config", () => {
            window.EdenConfig.styles = {
                family: [{ name: "Arial", value: "Arial, sans-serif" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const familyAction = richText.actions.find((a) => a.code === "family");
            expect(familyAction).toBeDefined();
        });
        it("should include font size styles from config", () => {
            window.EdenConfig.styles = {
                size: [{ name: "Large", value: "18px" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const sizeAction = richText.actions.find((a) => a.code === "size");
            expect(sizeAction).toBeDefined();
        });
        it("should include font weight styles from config", () => {
            window.EdenConfig.styles = {
                weight: [{ name: "Bold", value: "700" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const weightAction = richText.actions.find((a) => a.code === "weight");
            expect(weightAction).toBeDefined();
        });
    });
    describe("Actions", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should have bold action that executes bold command", () => {
            const boldAction = richText.actions.find((a) => a.code === "bold");
            expect(boldAction).toBeDefined();
            expect(typeof boldAction.action).toBe("function");
            boldAction.action();
            expect(document.execCommand).toHaveBeenCalledWith("bold");
        });
        it("should have italic action that executes italic command", () => {
            const italicAction = richText.actions.find((a) => a.code === "italic");
            expect(italicAction).toBeDefined();
            expect(typeof italicAction.action).toBe("function");
            italicAction.action();
            expect(document.execCommand).toHaveBeenCalledWith("italic");
        });
        it("should have list action that executes InsertUnorderedList command", () => {
            const listAction = richText.actions.find((a) => a.code === "list");
            expect(listAction).toBeDefined();
            expect(typeof listAction.action).toBe("function");
            listAction.action();
            expect(document.execCommand).toHaveBeenCalledWith("InsertUnorderedList");
        });
        it("should have orderedList action that executes InsertOrderedList command", () => {
            const orderedListAction = richText.actions.find((a) => a.code === "orderedList");
            expect(orderedListAction).toBeDefined();
            expect(typeof orderedListAction.action).toBe("function");
            orderedListAction.action();
            expect(document.execCommand).toHaveBeenCalledWith("InsertOrderedList");
        });
        it("should have left alignment action", () => {
            const leftAction = richText.actions.find((a) => a.code === "left");
            expect(leftAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof leftAction.action).toBe("function");
            leftAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("justifyleft");
        });
        it("should have center alignment action", () => {
            const centerAction = richText.actions.find((a) => a.code === "center");
            expect(centerAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof centerAction.action).toBe("function");
            centerAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("justifycenter");
        });
        it("should have right alignment action", () => {
            const rightAction = richText.actions.find((a) => a.code === "right");
            expect(rightAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof rightAction.action).toBe("function");
            rightAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("justifyright");
        });
        it("should have justify alignment action", () => {
            const justifyAction = richText.actions.find((a) => a.code === "justify");
            expect(justifyAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof justifyAction.action).toBe("function");
            justifyAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("justifyfull");
        });
        it("should have indent action", () => {
            const indentAction = richText.actions.find((a) => a.code === "indent");
            expect(indentAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof indentAction.action).toBe("function");
            indentAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("indent");
        });
        it("should have outdent action", () => {
            const outdentAction = richText.actions.find((a) => a.code === "outdent");
            expect(outdentAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof outdentAction.action).toBe("function");
            outdentAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("outdent");
        });
        it("should have cleanup action that removes formatting", () => {
            const cleanupAction = richText.actions.find((a) => a.code === "cleanup");
            expect(cleanupAction).toBeDefined();
            const focusSpy = vi.spyOn(element, "focus");
            expect(typeof cleanupAction.action).toBe("function");
            cleanupAction.action();
            expect(focusSpy).toHaveBeenCalled();
            expect(document.execCommand).toHaveBeenCalledWith("removeFormat");
        });
    });
    describe("updateLink", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const range = document.createRange();
            const textNode = document.createTextNode("test text");
            element.appendChild(textNode);
            range.selectNodeContents(textNode);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
        });
        it("should create link with url", () => {
            richText.updateLink("https://example.com");
            expect(document.execCommand).toHaveBeenCalledWith("createlink", false, "https://example.com");
        });
        it("should handle www prefix by adding https", () => {
            richText.updateLink("www.example.com");
            expect(document.execCommand).toHaveBeenCalledWith("createlink", false, "https://www.example.com");
        });
        it("should strip domain from internal urls", () => {
            Object.defineProperty(window, "location", {
                value: { protocol: "https:", host: "mysite.com" },
                writable: true,
                configurable: true,
            });
            const newRichText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            newRichText.updateLink("https://mysite.com/page");
            expect(document.execCommand).toHaveBeenCalledWith("createlink", false, "https://mysite.com/page");
        });
        it("should set link attributes after creation", () => {
            const anchor = document.createElement("a");
            anchor.textContent = "test text";
            element.innerHTML = "";
            element.appendChild(anchor);
            const range = document.createRange();
            range.selectNodeContents(anchor);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            richText.updateLink("https://example.com", "Test Title", true);
            expect(anchor.getAttribute("href")).toBe("https://example.com");
            expect(anchor.getAttribute("title")).toBe("Test Title");
            expect(anchor.getAttribute("target")).toBe("_blank");
            expect(anchor.getAttribute("rel")).toBe("noopener");
        });
        it("should remove attributes when not provided", () => {
            const anchor = document.createElement("a");
            anchor.setAttribute("href", "https://old.com");
            anchor.setAttribute("title", "Old Title");
            anchor.setAttribute("target", "_blank");
            anchor.setAttribute("rel", "noopener");
            anchor.textContent = "test text";
            element.innerHTML = "";
            element.appendChild(anchor);
            const range = document.createRange();
            range.selectNodeContents(anchor);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            richText.updateLink("", "", false);
            expect(anchor.hasAttribute("href")).toBe(false);
            expect(anchor.hasAttribute("title")).toBe(false);
            expect(anchor.hasAttribute("target")).toBe(false);
            expect(anchor.hasAttribute("rel")).toBe(false);
        });
    });
    describe("updateSource", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should update element innerHTML", () => {
            const html = "<p>New content</p>";
            richText.updateSource(html);
            expect(element.innerHTML).toBe(html);
        });
    });
    describe("cleanupEmptyWithBr", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should remove br when element is empty", () => {
            element.innerHTML = "<br>";
            richText.cleanupEmptyWithBr();
            expect(element.querySelector("br")).toBeNull();
        });
        it("should not remove br when element has content", () => {
            element.innerHTML = "Text<br>";
            richText.cleanupEmptyWithBr();
            expect(element.querySelector("br")).not.toBeNull();
        });
        it("should not remove br when multiple brs exist", () => {
            element.innerHTML = "<br><br>";
            richText.cleanupEmptyWithBr();
            expect(element.querySelectorAll("br").length).toBe(2);
        });
        it("should handle null element", () => {
            richText.element = null;
            expect(() => richText.cleanupEmptyWithBr()).not.toThrow();
        });
    });
    describe("activate", () => {
        it("should set element as editable and add event listeners", async () => {
            const brick = await import("../../core/brick.js");
            const setEditableSpy = vi.mocked(brick.setEditable);
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const addEventListenerSpy = vi.spyOn(element, "addEventListener");
            richText.activate();
            expect(setEditableSpy).toHaveBeenCalledWith(element);
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", richText.handleClick);
            expect(addEventListenerSpy).toHaveBeenCalledWith("input", richText.spawnOrUpdateEmailPresenceHelper);
            expect(addEventListenerSpy).toHaveBeenCalledWith("blur", richText.removeEmailPresenceHelper);
        });
        it("should add text length event listeners when option is set", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    textLength: { max: 100 },
                },
            });
            const addEventListenerSpy = vi.spyOn(element, "addEventListener");
            richText.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("input", richText.updateTextLengthHelper);
            expect(addEventListenerSpy).toHaveBeenCalledWith("focus", richText.spawnTextLengthHelper);
            expect(addEventListenerSpy).toHaveBeenCalledWith("blur", richText.removeTextLengthHelper);
        });
        it("should add padding when allowHtmlEdit is true", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    allowHtmlEdit: true,
                },
            });
            richText.activate();
            expect(element.style.padding).toMatch(/10px 0(px)?/);
        });
    });
    describe("desactivate", () => {
        it("should unset element as editable and remove event listeners", async () => {
            const brick = await import("../../core/brick.js");
            const unsetEditableSpy = vi.mocked(brick.unsetEditable);
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const removeEventListenerSpy = vi.spyOn(element, "removeEventListener");
            richText.desactivate();
            expect(unsetEditableSpy).toHaveBeenCalledWith(element);
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", richText.handleClick);
            expect(removeEventListenerSpy).toHaveBeenCalledWith("input", richText.spawnOrUpdateEmailPresenceHelper);
            expect(removeEventListenerSpy).toHaveBeenCalledWith("blur", richText.removeEmailPresenceHelper);
        });
        it("should remove text length event listeners when option is set", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    textLength: { max: 100 },
                },
            });
            const removeEventListenerSpy = vi.spyOn(element, "removeEventListener");
            richText.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("input", richText.updateTextLengthHelper);
            expect(removeEventListenerSpy).toHaveBeenCalledWith("focus", richText.spawnTextLengthHelper);
            expect(removeEventListenerSpy).toHaveBeenCalledWith("blur", richText.removeTextLengthHelper);
        });
        it("should remove style when allowHtmlEdit was enabled", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    allowHtmlEdit: true,
                },
            });
            element.style.padding = "10px 0";
            richText.desactivate();
            expect(element.hasAttribute("style")).toBe(false);
        });
        it("should cleanup empty br tags", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            element.innerHTML = "<br>";
            richText.desactivate();
            expect(element.querySelector("br")).toBeNull();
        });
    });
    describe("findParentFromSelection", () => {
        it("should find parent anchor from selection", () => {
            const anchor = document.createElement("a");
            anchor.href = "https://example.com";
            const textNode = document.createTextNode("link text");
            anchor.appendChild(textNode);
            element.appendChild(anchor);
            const range = document.createRange();
            range.selectNodeContents(textNode);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const parent = findParentFromSelection("a");
            expect(parent).toBe(anchor);
        });
        it("should return null when no parent matches", () => {
            const textNode = document.createTextNode("plain text");
            element.appendChild(textNode);
            const range = document.createRange();
            range.selectNodeContents(textNode);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const parent = findParentFromSelection("a");
            expect(parent).toBeNull();
        });
        it("should return null when no selection exists", () => {
            const selection = window.getSelection();
            selection?.removeAllRanges();
            const parent = findParentFromSelection("a");
            expect(parent).toBeNull();
        });
        it("should handle element node instead of text node", () => {
            const span = document.createElement("span");
            const textNode = document.createTextNode("text");
            span.appendChild(textNode);
            element.appendChild(span);
            const range = document.createRange();
            range.setStart(span, 0);
            range.setEnd(span, 1);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const parent = findParentFromSelection("span");
            expect(parent).toBe(span);
        });
        it("should not match contenteditable=true parent", () => {
            const contentEditable = document.createElement("div");
            contentEditable.setAttribute("contenteditable", "true");
            const textNode = document.createTextNode("text");
            contentEditable.appendChild(textNode);
            element.appendChild(contentEditable);
            const range = document.createRange();
            range.selectNodeContents(textNode);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const parent = findParentFromSelection("div");
            expect(parent).toBeNull();
        });
    });
    describe("Link action integration", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should have link action", () => {
            const linkAction = richText.actions.find((a) => a.code === "link");
            expect(linkAction).toBeDefined();
            expect(linkAction?.command).toBe("link");
        });
        it("should have unlink action", () => {
            const unlinkAction = richText.actions.find((a) => a.code === "unlink");
            expect(unlinkAction).toBeDefined();
        });
    });
    describe("Config-based style actions", () => {
        it("should create color action with formatter", () => {
            window.EdenConfig.styles = {
                color: [{ name: "Red", value: "#ff0000" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const colorAction = richText.actions.find((a) => a.code === "color");
            expect(colorAction).toBeDefined();
            const subAction = colorAction.action[0];
            expect(subAction.property).toBe("color");
            expect(subAction.name).toBe("Red");
            expect(subAction.value).toBe("#ff0000");
            expect(subAction.formatter).toBeDefined();
        });
        it("should create background action with formatter", () => {
            window.EdenConfig.styles = {
                background: [{ name: "Blue Background", value: "#0000ff" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const bgAction = richText.actions.find((a) => a.code === "background");
            expect(bgAction).toBeDefined();
            const subAction = bgAction.action[0];
            expect(subAction.property).toBe("background");
            expect(subAction.formatter).toBeDefined();
        });
        it("should create font family action without formatter", () => {
            window.EdenConfig.styles = {
                family: [{ name: "Arial", value: "Arial, sans-serif" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const familyAction = richText.actions.find((a) => a.code === "family");
            const subAction = familyAction.action[0];
            expect(subAction.property).toBe("font-family");
            expect(subAction.formatter).toBe(false);
        });
        it("should create font size action without formatter", () => {
            window.EdenConfig.styles = {
                size: [{ name: "16px", value: "16px" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const sizeAction = richText.actions.find((a) => a.code === "size");
            const subAction = sizeAction.action[0];
            expect(subAction.property).toBe("font-size");
            expect(subAction.formatter).toBe(false);
        });
        it("should create font weight action without formatter", () => {
            window.EdenConfig.styles = {
                weight: [{ name: "Bold", value: "700" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            const weightAction = richText.actions.find((a) => a.code === "weight");
            const subAction = weightAction.action[0];
            expect(subAction.property).toBe("font-weight");
            expect(subAction.formatter).toBe(false);
        });
    });
    describe("Edge cases", () => {
        it("should handle null element in activate", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            richText.element = null;
            expect(() => richText.activate()).not.toThrow();
        });
        it("should handle null element in desactivate", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            richText.element = null;
            expect(() => richText.desactivate()).not.toThrow();
        });
        it("should initialize with empty parentBricks array", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            expect(richText.parentBricks).toEqual([]);
        });
        it("should handle commands option as empty array", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    commands: [],
                },
            });
            expect(richText.actions.length).toBe(0);
        });
        it("should handle textLength option without force", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    textLength: {
                        max: 50,
                    },
                },
            });
            expect(richText.hasTextLengthOption).toBe(50);
            expect(richText.enforceTextLength).toBe(false);
        });
        it("should correctly set watchSelectionChange flag", () => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
            expect(richText.watchSelectionChange).toBe(true);
        });
    });
    describe("Action tooltips and icons", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should have proper tooltip for bold action", () => {
            const boldAction = richText.actions.find((a) => a.code === "bold");
            expect(boldAction?.tooltip).toContain("bricks.rich-text.bold");
        });
        it("should have proper icon for each action", () => {
            const actionCodes = [
                "bold",
                "italic",
                "list",
                "orderedList",
                "left",
                "center",
                "right",
            ];
            const icons = [
                "bold",
                "italic",
                "list",
                "ordered_list",
                "align_left",
                "align_center",
                "align_right",
            ];
            actionCodes.forEach((code, index) => {
                const action = richText.actions.find((a) => a.code === code);
                expect(action?.icon).toBe(icons[index]);
            });
        });
        it("should have command property for applicable actions", () => {
            const actionsWithCommands = ["bold", "italic", "list", "orderedList"];
            const commands = [
                "bold",
                "italic",
                "InsertUnorderedList",
                "InsertOrderedList",
            ];
            actionsWithCommands.forEach((code, index) => {
                const action = richText.actions.find((a) => a.code === code);
                expect(action?.command).toBe(commands[index]);
            });
        });
    });
    describe("Config style actions - style wrapping", () => {
        beforeEach(() => {
            window.EdenConfig.styles = {
                color: [{ name: "Red", value: "#ff0000" }],
            };
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should execute color action to wrap selection", () => {
            element.innerHTML = "Some text";
            const textNode = element.firstChild;
            const range = document.createRange();
            range.setStart(textNode, 0);
            range.setEnd(textNode, 4);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const colorAction = richText.actions.find((a) => a.code === "color");
            const redOption = (colorAction?.action)[0];
            redOption.action();
            expect(document.execCommand).toHaveBeenCalledWith("insertHTML", false, expect.stringContaining("color: #ff0000"));
        });
        it("should not wrap when selection is empty", () => {
            element.innerHTML = "Some text";
            const range = document.createRange();
            range.setStart(element.firstChild, 0);
            range.setEnd(element.firstChild, 0);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const colorAction = richText.actions.find((a) => a.code === "color");
            const redOption = (colorAction?.action)[0];
            const execCommandSpy = vi.spyOn(document, "execCommand");
            execCommandSpy.mockClear();
            redOption.action();
            expect(execCommandSpy).not.toHaveBeenCalledWith("insertHTML", expect.anything(), expect.anything());
        });
        it("should handle applying style to already styled element", () => {
            const span = document.createElement("span");
            span.style.color = "#ff0000";
            span.textContent = "Red text";
            element.appendChild(span);
            const range = document.createRange();
            range.selectNodeContents(span);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const colorAction = richText.actions.find((a) => a.code === "color");
            const redOption = (colorAction?.action)[0];
            redOption.action();
            expect(element.innerHTML).toContain("Red text");
        });
        it("should handle element nodes in wrapWith", () => {
            const strong = document.createElement("strong");
            strong.textContent = "Bold text";
            element.appendChild(strong);
            const range = document.createRange();
            range.setStart(strong.firstChild, 0);
            range.setEnd(strong.firstChild, 4);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const colorAction = richText.actions.find((a) => a.code === "color");
            const redOption = (colorAction?.action)[0];
            redOption.action();
            expect(document.execCommand).toHaveBeenCalledWith("insertHTML", false, expect.stringContaining("<strong>"));
        });
        it("should handle no selection gracefully", () => {
            const selection = window.getSelection();
            selection?.removeAllRanges();
            const colorAction = richText.actions.find((a) => a.code === "color");
            const redOption = (colorAction?.action)[0];
            try {
                redOption.action();
            }
            catch (e) {
                expect(e).toBeDefined();
            }
        });
    });
    describe("Link and unlink actions", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should execute unlink when anchor is selected", () => {
            const anchor = document.createElement("a");
            anchor.href = "https://example.com";
            anchor.textContent = "link";
            element.appendChild(anchor);
            const range = document.createRange();
            range.selectNodeContents(anchor);
            const selection = window.getSelection();
            selection?.removeAllRanges();
            selection?.addRange(range);
            const unlinkAction = richText.actions.find((a) => a.code === "unlink");
            expect(typeof unlinkAction.action).toBe("function");
            unlinkAction.action();
            expect(document.execCommand).toHaveBeenCalledWith("unlink", false, "false");
        });
    });
    describe("Async actions - link modal", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
            });
        });
        it("should have link action that opens modal", async () => {
            const linkAction = richText.actions.find((a) => a.code === "link");
            expect(linkAction).toBeDefined();
            expect(typeof linkAction?.action).toBe("function");
        });
    });
    describe("Async actions - source modal", () => {
        beforeEach(() => {
            richText = new RichTextFormat({
                block: block,
                element,
                parentBricks: [],
                options: {
                    allowHtmlEdit: true,
                },
            });
        });
        it("should have source action when allowHtmlEdit is enabled", () => {
            const sourceAction = richText.actions.find((a) => a.code === "source");
            expect(sourceAction).toBeDefined();
            expect(typeof sourceAction?.action).toBe("function");
        });
    });
});
