import { describe, it, expect, beforeEach, vi, afterEach } from "vitest";
import Recursive from "../Recursive";
import { snackbar } from "../../user-interface/common/eden-snackbar";
import EdenModal from "../../user-interface/common/eden-modal";
vi.mock("../../user-interface/common/eden-snackbar", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-modal", () => {
    const MockModal = vi.fn().mockImplementation(function () {
        return document.createElement("div");
    });
    return {
        default: MockModal,
    };
});
vi.mock("../../i18n/index", () => ({
    default: vi.fn((key) => key),
}));
vi.mock("../../services/type-of", () => ({
    default: vi.fn((value) => {
        if (value === undefined)
            return "undefined";
        if (typeof value === "string")
            return "string";
        if (typeof value === "object")
            return "object";
        return typeof value;
    }),
}));
vi.mock("../../services/put", () => ({
    put: vi.fn((element, selector) => {
        const parts = selector.split(" > ");
        let current = element;
        for (const part of parts) {
            const cleanSelector = part.trim();
            if (!cleanSelector)
                continue;
            let child = current.querySelector(cleanSelector);
            if (!child) {
                child = document.createElement("div");
                if (cleanSelector.startsWith(".")) {
                    child.classList.add(cleanSelector.substring(1));
                }
                else if (cleanSelector.startsWith("#")) {
                    child.id = cleanSelector.substring(1);
                }
                current.appendChild(child);
            }
            current = child;
        }
        return current;
    }),
}));
vi.mock("../../services/html", () => ({
    htmlStringToElements: vi.fn((html) => {
        const div = document.createElement("div");
        div.innerHTML = html;
        return Array.from(div.children);
    }),
}));
vi.mock("../Text", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../RichText", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Picture", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Link", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Icon", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Slideshow", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Table", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Video", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Iframe", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../GoogleMaps", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../YoutubePlaylist", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
vi.mock("../Spotify", () => ({
    default: vi
        .fn()
        .mockImplementation(function ({ block, element, parentBricks }) {
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
        this.init = vi.fn();
        this.activate = vi.fn();
    }),
}));
class MockBrick {
    constructor({ block, element, parentBricks }) {
        this.init = vi.fn();
        this.activate = vi.fn();
        this.element = element;
        this.block = block;
        this.parentBricks = parentBricks || [];
    }
}
describe("Recursive Brick", () => {
    let mockBlock;
    let element;
    let templates;
    beforeEach(() => {
        element = document.createElement("div");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
            instances: [],
        };
        templates = [
            {
                id: "template-1",
                name: "Template 1",
                markup: {
                    ".title": {
                        type: "Text",
                        markup: "",
                    },
                },
            },
            {
                id: "template-2",
                name: "Template 2",
                markup: {
                    ".content": {
                        type: "RichText",
                        markup: "",
                    },
                },
            },
        ];
    });
    afterEach(() => {
        vi.clearAllMocks();
    });
    describe("constructor", () => {
        it("should initialize Recursive brick with templates", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            expect(recursive.element).toBe(element);
            expect(recursive.block).toBe(mockBlock);
            expect(recursive.templates).toBe(templates);
            expect(recursive.container).toBe(element);
            expect(recursive.actions).toHaveLength(5);
        });
        it("should throw error if no templates provided", () => {
            expect(() => {
                new Recursive({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                    templates: [],
                });
            }).toThrow("No templates specified for Recursive brick");
        });
        it("should throw error if templates is undefined", () => {
            expect(() => {
                new Recursive({
                    block: mockBlock,
                    element,
                    parentBricks: [],
                });
            }).toThrow("No templates specified for Recursive brick");
        });
        it("should create actions with correct structure", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            expect(recursive.actions[0]).toHaveProperty("icon", "plus");
            expect(recursive.actions[0]).toHaveProperty("tooltip");
            expect(Array.isArray(recursive.actions[0].action)).toBe(true);
            expect(recursive.actions[1]).toHaveProperty("icon", "duplicate");
            expect(recursive.actions[2]).toHaveProperty("icon", "chevron_up");
            expect(recursive.actions[3]).toHaveProperty("icon", "chevron_down");
            expect(recursive.actions[4]).toHaveProperty("icon", "delete");
        });
        it("should map templates to add actions", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const addActions = recursive.actions[0].action;
            expect(addActions).toHaveLength(2);
            expect(addActions[0]).toHaveProperty("name", "Template 1");
            expect(addActions[1]).toHaveProperty("name", "Template 2");
        });
    });
    describe("addItem", () => {
        it("should add item to empty container", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.addItem(templates[0]);
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(1);
            expect(items[0].getAttribute("data-recursive-id")).toBe("template-1");
        });
        it("should add item after selected item", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const firstItem = document.createElement("div");
            firstItem.dataset.recursiveId = "template-1";
            element.appendChild(firstItem);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(firstItem);
            await recursive.addItem(templates[1]);
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(2);
            expect(items[1].getAttribute("data-recursive-id")).toBe("template-2");
        });
        it("should create DOM structure from markup", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.addItem(templates[0]);
            const item = element.querySelector("[data-recursive-id='template-1']");
            expect(item).not.toBeNull();
        });
        it("should initialize nested bricks", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.addItem(templates[0]);
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(1);
        });
        it("should handle template without markup", async () => {
            const templateNoMarkup = {
                id: "no-markup",
                name: "No Markup",
                markup: {},
            };
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates: [templateNoMarkup],
            });
            await recursive.addItem(templateNoMarkup);
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(1);
        });
        it("should include Recursive brick in parent chain", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.addItem(templates[0]);
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(1);
        });
    });
    describe("duplicateItem", () => {
        it("should show snackbar when no item selected", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(null);
            await recursive.duplicateItem();
            expect(snackbar).toHaveBeenCalledWith("common.no-selection", "information");
        });
        it("should clone selected item", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "template-1";
            item.innerHTML = "<p>Test content</p>";
            element.appendChild(item);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item);
            await recursive.duplicateItem();
            const items = element.querySelectorAll("[data-recursive-id='template-1']");
            expect(items).toHaveLength(2);
            expect(items[1].innerHTML).toContain("Test content");
        });
        it("should remove contenteditable from cloned elements", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "template-1";
            const editableDiv = document.createElement("div");
            editableDiv.contentEditable = "true";
            item.appendChild(editableDiv);
            element.appendChild(item);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item);
            await recursive.duplicateItem();
            const items = element.querySelectorAll("[data-recursive-id='template-1']");
            const clonedEditable = items[1].querySelector("[contenteditable]");
            expect(clonedEditable).toBeNull();
        });
        it("should insert clone after selected item", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "template-1";
            item1.textContent = "First";
            element.appendChild(item1);
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "template-2";
            item2.textContent = "Second";
            element.appendChild(item2);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item1);
            await recursive.duplicateItem();
            const items = Array.from(element.children);
            expect(items[1].textContent).toBe("First");
        });
        it("should re-initialize bricks in cloned item", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "template-1";
            element.appendChild(item);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item);
            await recursive.duplicateItem();
            const items = element.querySelectorAll("[data-recursive-id='template-1']");
            expect(items.length).toBe(2);
        });
    });
    describe("moveItem", () => {
        it("should show snackbar when no item selected", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(null);
            recursive.moveItem(-1);
            expect(snackbar).toHaveBeenCalledWith("common.no-selection", "information");
        });
        it("should move item up", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "item1";
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "item2";
            element.appendChild(item1);
            element.appendChild(item2);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item2);
            recursive.moveItem(-1);
            const items = Array.from(element.children);
            expect(items[0]).toBe(item2);
            expect(items[1]).toBe(item1);
        });
        it("should move item down", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "item1";
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "item2";
            element.appendChild(item1);
            element.appendChild(item2);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item1);
            recursive.moveItem(1);
            const items = Array.from(element.children);
            expect(items[0]).toBe(item2);
            expect(items[1]).toBe(item1);
        });
        it("should show warning when trying to move first item up", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "item1";
            element.appendChild(item);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item);
            recursive.moveItem(-1);
            expect(snackbar).toHaveBeenCalledWith("bricks.recursive.move-up", "warning");
        });
        it("should show warning when trying to move last item down", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "item1";
            element.appendChild(item);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item);
            recursive.moveItem(1);
            expect(snackbar).toHaveBeenCalledWith("bricks.recursive.move-down", "warning");
        });
    });
    describe("removeItem", () => {
        it("should show snackbar when no item selected", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(null);
            recursive.removeItem();
            expect(snackbar).toHaveBeenCalledWith("common.no-selection", "information");
        });
        it("should prevent deletion of last item", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "item1";
            element.appendChild(item);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item);
            recursive.removeItem();
            expect(snackbar).toHaveBeenCalledWith("bricks.recursive.cannot-remove-last-item", "warning");
            expect(element.children).toHaveLength(1);
        });
        it("should show confirmation modal for deletion", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "item1";
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "item2";
            element.appendChild(item1);
            element.appendChild(item2);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item1);
            recursive.removeItem();
            expect(EdenModal).toHaveBeenCalled();
        });
        it("should remove item when confirmed", () => {
            let confirmCallback;
            EdenModal.mockImplementation(function ({ buttons }) {
                confirmCallback = buttons[1].onClick;
                return document.createElement("div");
            });
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "item1";
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "item2";
            element.appendChild(item1);
            element.appendChild(item2);
            vi.spyOn(recursive, "getSelectedItem").mockReturnValue(item1);
            recursive.removeItem();
            if (confirmCallback) {
                confirmCallback();
            }
            expect(element.children).toHaveLength(1);
            expect(element.children[0]).toBe(item2);
        });
    });
    describe("getSelectedItem", () => {
        it("should return null when no selection", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            vi.spyOn(window, "getSelection").mockReturnValue(null);
            const result = recursive.getSelectedItem();
            expect(result).toBeNull();
        });
        it("should return null when selection has no ranges", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const mockSelection = {
                rangeCount: 0,
                getRangeAt: vi.fn(),
            };
            vi.spyOn(window, "getSelection").mockReturnValue(mockSelection);
            const result = recursive.getSelectedItem();
            expect(result).toBeNull();
        });
        it("should find item with recursiveId in parent chain", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "item1";
            const child = document.createElement("p");
            item.appendChild(child);
            element.appendChild(item);
            const range = document.createRange();
            range.selectNode(child);
            const mockSelection = {
                rangeCount: 1,
                getRangeAt: vi.fn(() => range),
            };
            vi.spyOn(window, "getSelection").mockReturnValue(mockSelection);
            const result = recursive.getSelectedItem();
            expect(result).toBe(item);
        });
        it("should return null when no recursiveId found", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const child = document.createElement("p");
            element.appendChild(child);
            const range = document.createRange();
            range.selectNode(child);
            const mockSelection = {
                rangeCount: 1,
                getRangeAt: vi.fn(() => range),
            };
            vi.spyOn(window, "getSelection").mockReturnValue(mockSelection);
            const result = recursive.getSelectedItem();
            expect(result).toBeNull();
        });
        it("should handle text node selection", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const item = document.createElement("div");
            item.dataset.recursiveId = "item1";
            const textNode = document.createTextNode("Test text");
            item.appendChild(textNode);
            element.appendChild(item);
            const range = document.createRange();
            range.selectNode(textNode);
            const mockSelection = {
                rangeCount: 1,
                getRangeAt: vi.fn(() => range),
            };
            vi.spyOn(window, "getSelection").mockReturnValue(mockSelection);
            const result = recursive.getSelectedItem();
            expect(result).toBe(item);
        });
    });
    describe("init", () => {
        it("should initialize existing items", async () => {
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "template-1";
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "template-2";
            element.appendChild(item1);
            element.appendChild(item2);
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.init();
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(2);
        });
        it("should auto-generate first item when container is empty", async () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const addItemSpy = vi.spyOn(recursive, "addItem");
            await recursive.init();
            expect(addItemSpy).toHaveBeenCalledWith(templates[0]);
        });
        it("should not activate bricks for existing items", async () => {
            const item = document.createElement("div");
            item.dataset.recursiveId = "template-1";
            element.appendChild(item);
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.init();
            const createdBricks = mockBlock.instances;
            createdBricks.forEach((brick) => {
            });
        });
        it("should handle multiple existing items", async () => {
            const item1 = document.createElement("div");
            item1.dataset.recursiveId = "template-1";
            const item2 = document.createElement("div");
            item2.dataset.recursiveId = "template-1";
            const item3 = document.createElement("div");
            item3.dataset.recursiveId = "template-2";
            element.appendChild(item1);
            element.appendChild(item2);
            element.appendChild(item3);
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await recursive.init();
            const items = element.querySelectorAll("[data-recursive-id]");
            expect(items).toHaveLength(3);
        });
        it("should handle item with unknown template id", async () => {
            const item = document.createElement("div");
            item.dataset.recursiveId = "unknown-template";
            element.appendChild(item);
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            await expect(recursive.init()).resolves.not.toThrow();
        });
    });
    describe("activate", () => {
        it("should add click event listener", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const addEventListenerSpy = vi.spyOn(element, "addEventListener");
            recursive.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", recursive.handleClick);
        });
        it("should handle null container", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            recursive.container = null;
            expect(() => recursive.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const removeEventListenerSpy = vi.spyOn(element, "removeEventListener");
            recursive.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", recursive.handleClick);
        });
        it("should handle null container", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            recursive.container = null;
            expect(() => recursive.desactivate()).not.toThrow();
        });
    });
    describe("handleClick", () => {
        it("should call block.manageActions with correct brick chain", () => {
            const parentBrick = new MockBrick({
                block: mockBlock,
                element: document.createElement("div"),
                parentBricks: [],
            });
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [parentBrick],
                templates,
            });
            const event = new MouseEvent("click");
            const preventDefaultSpy = vi.spyOn(event, "preventDefault");
            const stopPropagationSpy = vi.spyOn(event, "stopPropagation");
            recursive.handleClick(event);
            expect(preventDefaultSpy).toHaveBeenCalled();
            expect(stopPropagationSpy).toHaveBeenCalled();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([
                parentBrick,
                recursive,
            ]);
        });
        it("should call block.manageActions with empty parent chain", () => {
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates,
            });
            const event = new MouseEvent("click");
            recursive.handleClick(event);
            expect(mockBlock.manageActions).toHaveBeenCalledWith([recursive]);
        });
        it("should handle handleClick with null block", () => {
            const recursive = new Recursive({
                block: null,
                element,
                parentBricks: [],
                templates,
            });
            const event = new MouseEvent("click");
            expect(() => recursive.handleClick(event)).not.toThrow();
        });
    });
    describe("edge cases and error handling", () => {
        it("should handle template with nested objects", async () => {
            const complexTemplate = {
                id: "complex",
                name: "Complex",
                markup: {
                    ".header": {
                        ".title": {
                            type: "Text",
                            markup: "",
                        },
                    },
                },
            };
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates: [complexTemplate],
            });
            await expect(recursive.addItem(complexTemplate)).resolves.not.toThrow();
        });
        it("should handle template with string markup", async () => {
            const stringMarkupTemplate = {
                id: "string-markup",
                name: "String Markup",
                markup: {
                    ".content": {
                        markup: "<p>HTML content</p>",
                    },
                },
            };
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates: [stringMarkupTemplate],
            });
            await expect(recursive.addItem(stringMarkupTemplate)).resolves.not.toThrow();
        });
        it("should handle Table brick with special parameters", async () => {
            const tableTemplate = {
                id: "table-template",
                name: "Table",
                markup: {
                    ".table": {
                        type: "Table",
                        markup: [["cell1", "cell2"]],
                    },
                },
            };
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates: [tableTemplate],
            });
            try {
                await recursive.addItem(tableTemplate);
                expect(true).toBe(true);
            }
            catch (error) {
                expect(error).toBeDefined();
            }
        });
        it("should handle missing block gracefully", async () => {
            const recursive = new Recursive({
                block: undefined,
                element,
                parentBricks: [],
                templates,
            });
            await expect(recursive.addItem(templates[0])).resolves.not.toThrow();
        });
        it("should handle brick initialization failure", async () => {
            const failingTemplate = {
                id: "failing",
                name: "Failing",
                markup: {
                    ".content": {
                        type: "NonExistentBrick",
                        markup: "",
                    },
                },
            };
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates: [failingTemplate],
            });
            await expect(recursive.addItem(failingTemplate)).resolves.not.toThrow();
        });
        it("should handle multiple actions for single template", () => {
            const singleTemplate = [templates[0]];
            const recursive = new Recursive({
                block: mockBlock,
                element,
                parentBricks: [],
                templates: singleTemplate,
            });
            const addActions = recursive.actions[0].action;
            expect(addActions).toHaveLength(1);
            expect(addActions[0].name).toBe("Template 1");
        });
    });
    describe("integration scenarios", () => {
        it("should maintain brick parent chain through operations", async () => {
            const testElement = document.createElement("div");
            const testBlock = {
                obsolete: false,
                element: document.createElement("section"),
                manageActions: vi.fn(),
                instances: [],
                init: vi.fn(),
                activate: vi.fn(),
                desactivate: vi.fn(),
                cleanRefs: vi.fn(),
            };
            const parentBrick = new MockBrick({
                block: testBlock,
                element: testElement,
                parentBricks: [],
            });
            const recursive = new Recursive({
                block: testBlock,
                element: testElement,
                parentBricks: [parentBrick],
                templates,
            });
            const addItemSpy = vi.spyOn(recursive, "addItem");
            await recursive.addItem(templates[0]);
            expect(addItemSpy).toHaveBeenCalledWith(templates[0]);
            expect(recursive.parentBricks).toContain(parentBrick);
        });
    });
});
