import { describe, it, expect, beforeEach, vi } from "vitest";
import Picture from "../Picture";
vi.mock("../../user-interface/common/eden-modal", () => ({
    default: class MockEdenModal {
        constructor(config) {
            this.config = config;
        }
    },
}));
describe("Picture Brick", () => {
    let mockBlock;
    let element;
    let img;
    beforeEach(() => {
        element = document.createElement("div");
        img = document.createElement("img");
        img.src = "test.jpg";
        img.alt = "Test image";
        element.appendChild(img);
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
    });
    describe("constructor", () => {
        it("should initialize Picture brick", () => {
            const picture = new Picture({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(picture.element).toBe(element);
            expect(picture.image).toBe(img);
            expect(picture.actions.length).toBeGreaterThan(0);
        });
        it("should find img element in container", () => {
            const picture = new Picture({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(picture.image?.tagName).toBe("IMG");
            expect(picture.image?.src).toContain("test.jpg");
        });
        it("should handle missing img element", () => {
            const emptyElement = document.createElement("div");
            const picture = new Picture({
                block: mockBlock,
                element: emptyElement,
                parentBricks: [],
            });
            expect(picture.image).toBeNull();
        });
        it("should store sizes options", () => {
            const options = {
                sizes: {
                    small: { src: "small.jpg", width: 400, height: 300 },
                    large: { src: "large.jpg", width: 800, height: 600 },
                },
            };
            const picture = new Picture({
                block: mockBlock,
                element,
                options,
                parentBricks: [],
            });
            expect(picture.sizes).toEqual(options.sizes);
        });
        it("should store markup", () => {
            const markup = "<img src='default.jpg' alt='Default'>";
            const picture = new Picture({
                block: mockBlock,
                element,
                markup,
                parentBricks: [],
            });
            expect(picture.markup).toBe(markup);
        });
    });
    describe("buildActions", () => {
        it("should build base actions", () => {
            const picture = new Picture({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(picture.actions.length).toBeGreaterThan(0);
            const replaceAction = picture.actions.find((action) => action.icon === "image");
            expect(replaceAction).toBeDefined();
        });
        it("should include settings action", () => {
            const picture = new Picture({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const settingsAction = picture.actions.find((action) => action.icon === "tune");
            expect(settingsAction).toBeDefined();
        });
        it("should include remove action", () => {
            const picture = new Picture({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const removeAction = picture.actions.find((action) => action.icon === "close");
            expect(removeAction).toBeDefined();
        });
        it("should include crop action when metadata exists", () => {
            img.setAttribute("data-metadata", "{}");
            const picture = new Picture({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const cropAction = picture.actions.find((action) => action.icon === "crop");
            expect(cropAction).toBeDefined();
        });
        it("should not include crop action with sizes option", () => {
            img.setAttribute("data-metadata", "{}");
            const options = {
                sizes: {
                    small: { src: "small.jpg", width: 400, height: 300 },
                },
            };
            const picture = new Picture({
                block: mockBlock,
                element,
                options,
                parentBricks: [],
            });
            const cropAction = picture.actions.find((action) => action.icon === "crop");
            expect(cropAction).toBeUndefined();
        });
    });
});
