import { describe, it, expect, beforeEach, vi } from "vitest";
import Link from "../Link";
describe("Link Brick", () => {
    let mockBlock;
    let element;
    beforeEach(() => {
        element = document.createElement("a");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        Object.defineProperty(window, "location", {
            value: {
                protocol: "https:",
                host: "mysite.com",
                href: "https://mysite.com",
            },
            writable: true,
            configurable: true,
        });
    });
    describe("constructor", () => {
        it("should initialize Link brick", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(link.element).toBe(element);
            expect(link.block).toBe(mockBlock);
            expect(link.actions).toHaveLength(1);
            expect(link.actions[0].icon).toBe("link");
        });
        it("should initialize element with default options on creation", () => {
            const options = {
                defaultHref: "https://example.com",
                defaultTitle: "Example",
                defaultTarget: "_blank",
            };
            new Link({
                block: mockBlock,
                element,
                options,
                isCreation: true,
                parentBricks: [],
            });
            expect(element.href).toContain("example.com");
            expect(element.title).toBe("Example");
            expect(element.target).toBe("_blank");
            expect(element.getAttribute("rel")).toBe("noopener");
        });
        it("should not initialize element without isCreation flag", () => {
            const options = {
                defaultHref: "https://example.com",
                defaultTitle: "Example",
            };
            new Link({
                block: mockBlock,
                element,
                options,
                isCreation: false,
                parentBricks: [],
            });
            expect(element.href).not.toContain("example.com");
        });
    });
    describe("initElement", () => {
        it("should set href from options", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.initElement({ defaultHref: "https://test.com" });
            expect(element.href).toContain("test.com");
        });
        it("should set title from options", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.initElement({ defaultTitle: "Test Title" });
            expect(element.title).toBe("Test Title");
        });
        it("should set target and rel for _blank", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.initElement({ defaultTarget: "_blank" });
            expect(element.target).toBe("_blank");
            expect(element.getAttribute("rel")).toBe("noopener");
        });
        it("should handle missing options gracefully", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => link.initElement({})).not.toThrow();
        });
    });
    describe("updateLink", () => {
        it("should update href attribute", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://example.com");
            expect(element.hasAttribute("href")).toBe(true);
        });
        it("should remove protocol and host from internal URLs", () => {
            Object.defineProperty(window, "location", {
                value: {
                    protocol: "https:",
                    host: "mysite.com",
                },
                writable: true,
                configurable: true,
            });
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://mysite.com/page");
            expect(element.getAttribute("href")).toBe("/page");
        });
        it("should add https protocol to www URLs", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("www.example.com");
            expect(element.getAttribute("href")).toBe("https://www.example.com");
        });
        it("should remove href attribute if URL is empty", () => {
            element.setAttribute("href", "https://example.com");
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("");
            expect(element.hasAttribute("href")).toBe(false);
        });
        it("should update title attribute", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://example.com", "Example Title");
            expect(element.getAttribute("title")).toBe("Example Title");
        });
        it("should remove title attribute if empty", () => {
            element.setAttribute("title", "Old Title");
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://example.com", "");
            expect(element.hasAttribute("title")).toBe(false);
        });
        it("should set target and rel for external links", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://example.com", undefined, true);
            expect(element.getAttribute("target")).toBe("_blank");
            expect(element.getAttribute("rel")).toBe("noopener");
        });
        it("should remove target and rel if not external", () => {
            element.setAttribute("target", "_blank");
            element.setAttribute("rel", "noopener");
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://example.com", undefined, false);
            expect(element.hasAttribute("target")).toBe(false);
            expect(element.hasAttribute("rel")).toBe(false);
        });
        it("should handle all parameters together", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.updateLink("https://example.com", "Example", true);
            expect(element.getAttribute("href")).toBe("https://example.com");
            expect(element.getAttribute("title")).toBe("Example");
            expect(element.getAttribute("target")).toBe("_blank");
            expect(element.getAttribute("rel")).toBe("noopener");
        });
    });
    describe("activate and desactivate", () => {
        it("should add click event listener on activate", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const spy = vi.spyOn(element, "addEventListener");
            link.activate();
            expect(spy).toHaveBeenCalledWith("click", link.handleClick);
        });
        it("should remove click event listener on desactivate", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const spy = vi.spyOn(element, "removeEventListener");
            link.desactivate();
            expect(spy).toHaveBeenCalledWith("click", link.handleClick);
        });
        it("should not throw if element is null", () => {
            const link = new Link({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            link.element = null;
            expect(() => link.activate()).not.toThrow();
            expect(() => link.desactivate()).not.toThrow();
        });
    });
});
