import { describe, it, expect, beforeEach, vi, afterEach } from "vitest";
import Iframe from "../Iframe";
vi.mock("../../services/url", () => ({
    parseUrl: vi.fn(),
}));
vi.mock("../../user-interface/plugin/iframe-modal", () => ({
    default: vi.fn(),
}));
vi.mock("../../user-interface/common/eden-snackbar", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../../i18n/index", () => ({
    default: vi.fn((key) => key),
}));
import { parseUrl } from "../../services/url";
import { snackbar } from "../../user-interface/common/eden-snackbar";
import IframeModal from "../../user-interface/plugin/iframe-modal";
describe("Iframe Brick", () => {
    let mockBlock;
    let element;
    let parseUrlMock;
    let snackbarMock;
    beforeEach(() => {
        element = document.createElement("div");
        mockBlock = {
            obsolete: false,
            element: document.createElement("section"),
            manageActions: vi.fn(),
        };
        parseUrlMock = vi.mocked(parseUrl);
        snackbarMock = vi.mocked(snackbar);
        vi.clearAllMocks();
    });
    afterEach(() => {
        element.innerHTML = "";
    });
    describe("constructor", () => {
        it("should initialize Iframe brick with empty element", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(iframe.element).toBe(element);
            expect(iframe.block).toBe(mockBlock);
            expect(iframe.mask).toBeInstanceOf(HTMLDivElement);
        });
        it("should create mask div with correct styles", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(iframe.mask.style.position).toBe("absolute");
            expect(iframe.mask.style.top).toBe("0px");
            expect(iframe.mask.style.bottom).toBe("0px");
            expect(iframe.mask.style.left).toBe("0px");
            expect(iframe.mask.style.right).toBe("0px");
        });
        it("should initialize actions with insert-content action", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(iframe.actions).toHaveLength(1);
            expect(iframe.actions[0].tooltip).toBe("bricks.iframe.insert-content");
            expect(iframe.actions[0].icon).toBe("source");
            expect(typeof iframe.actions[0].action).toBe("function");
        });
        it("should handle element with existing iframe", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://example.com";
            existingIframe.setAttribute("width", "800");
            existingIframe.setAttribute("height", "600");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(iframe.element.querySelector("iframe")).toBe(existingIframe);
        });
        it("should store parent bricks", () => {
            const parentBricks = [
                new Iframe({ block: mockBlock, element, parentBricks: [] }),
            ];
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks,
            });
            expect(iframe.parentBricks).toBe(parentBricks);
        });
    });
    describe("actions", () => {
        it("should call IframeModal with empty values for new iframe", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.actions[0].action();
            expect(IframeModal).toHaveBeenCalledWith(iframe.updateIframe, "", "1000", "450", false);
        });
        it("should call IframeModal with existing iframe values", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://example.com/video";
            existingIframe.setAttribute("width", "800");
            existingIframe.setAttribute("height", "600");
            existingIframe.setAttribute("allow", "fullscreen");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.actions[0].action();
            expect(IframeModal).toHaveBeenCalledWith(iframe.updateIframe, "https://example.com/video", "800", "600", true);
        });
        it("should use default values when iframe attributes are missing", () => {
            const existingIframe = document.createElement("iframe");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.actions[0].action();
            expect(IframeModal).toHaveBeenCalledWith(iframe.updateIframe, "", "1000", "450", false);
        });
        it("should handle iframe without fullscreen attribute", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://example.com";
            existingIframe.setAttribute("width", "800");
            existingIframe.setAttribute("height", "600");
            existingIframe.setAttribute("allow", "autoplay");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.actions[0].action();
            expect(IframeModal).toHaveBeenCalledWith(iframe.updateIframe, "https://example.com", "800", "600", false);
        });
    });
    describe("updateIframe", () => {
        it("should create new iframe with valid URL", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
                protocol: "https:",
                hostname: "example.com",
                port: "",
                path: "/embed",
                search: "",
                hash: "",
                searchObject: {},
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "800", "600", false);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe).toBeTruthy();
            expect(createdIframe?.getAttribute("src")).toBe("https://example.com/embed");
            expect(createdIframe?.getAttribute("width")).toBe("800");
            expect(createdIframe?.getAttribute("height")).toBe("600");
            expect(createdIframe?.hasAttribute("allow")).toBe(false);
        });
        it("should set iframe style to max-width 100%", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "800", "600", false);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe?.style.maxWidth).toBe("100%");
        });
        it("should create iframe element properly", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "800", "600", false);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe).toBeTruthy();
            expect(createdIframe?.tagName).toBe("IFRAME");
        });
        it("should add fullscreen allow attribute when enabled", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "800", "600", true);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("allow")).toBe("fullscreen");
        });
        it("should remove allow attribute when fullscreen is disabled", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "800", "600", true);
            let createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("allow")).toBe("fullscreen");
            iframe.updateIframe("https://example.com/embed", "800", "600", false);
            createdIframe = element.querySelector("iframe");
            expect(createdIframe?.hasAttribute("allow")).toBe(false);
        });
        it("should update existing iframe", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
            });
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://old.com";
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "1200", "800", true);
            const updatedIframe = element.querySelector("iframe");
            expect(updatedIframe).toBe(existingIframe);
            expect(updatedIframe?.getAttribute("src")).toBe("https://example.com/embed");
            expect(updatedIframe?.getAttribute("width")).toBe("1200");
            expect(updatedIframe?.getAttribute("height")).toBe("800");
            expect(updatedIframe?.getAttribute("allow")).toBe("fullscreen");
        });
        it("should append mask when creating new iframe", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/embed",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/embed", "800", "600", false);
            expect(element.contains(iframe.mask)).toBe(true);
        });
        it("should show error snackbar for invalid URL", () => {
            parseUrlMock.mockReturnValue(null);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("not-a-valid-url", "800", "600", false);
            expect(snackbarMock).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
            expect(element.querySelector("iframe")).toBeNull();
        });
        it("should show error snackbar when URL has no host", () => {
            parseUrlMock.mockReturnValue({
                url: "/relative/path",
                host: null,
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("/relative/path", "800", "600", false);
            expect(snackbarMock).toHaveBeenCalledWith("snackbars.fill-valid-url", "error");
            expect(element.querySelector("iframe")).toBeNull();
        });
        it("should handle different width and height values", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com", "100%", "auto", false);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("width")).toBe("100%");
            expect(createdIframe?.getAttribute("height")).toBe("auto");
        });
    });
    describe("activate", () => {
        it("should add click event listener", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const addEventListenerSpy = vi.spyOn(element, "addEventListener");
            iframe.activate();
            expect(addEventListenerSpy).toHaveBeenCalledWith("click", iframe.handleClick);
        });
        it("should set position relative and append mask when iframe exists", () => {
            const existingIframe = document.createElement("iframe");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.activate();
            expect(element.style.position).toBe("relative");
            expect(element.contains(iframe.mask)).toBe(true);
        });
        it("should not append mask when no iframe exists", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.activate();
            expect(element.contains(iframe.mask)).toBe(false);
        });
        it("should handle null element gracefully", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.element = null;
            expect(() => iframe.activate()).not.toThrow();
        });
    });
    describe("desactivate", () => {
        it("should remove click event listener", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const removeEventListenerSpy = vi.spyOn(element, "removeEventListener");
            iframe.desactivate();
            expect(removeEventListenerSpy).toHaveBeenCalledWith("click", iframe.handleClick);
        });
        it("should remove mask element", () => {
            const existingIframe = document.createElement("iframe");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.activate();
            expect(element.contains(iframe.mask)).toBe(true);
            iframe.desactivate();
            expect(element.contains(iframe.mask)).toBe(false);
        });
        it("should remove style attribute from element", () => {
            const existingIframe = document.createElement("iframe");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.activate();
            expect(element.hasAttribute("style")).toBe(true);
            iframe.desactivate();
            expect(element.hasAttribute("style")).toBe(false);
        });
        it("should handle null element gracefully", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.element = null;
            expect(() => iframe.desactivate()).not.toThrow();
        });
    });
    describe("handleClick", () => {
        it("should prevent default and stop propagation", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            const mockEvent = {
                preventDefault: vi.fn(),
                stopPropagation: vi.fn(),
            };
            iframe.handleClick(mockEvent);
            expect(mockEvent.preventDefault).toHaveBeenCalled();
            expect(mockEvent.stopPropagation).toHaveBeenCalled();
        });
        it("should call block.manageActions with self and parent bricks", () => {
            const parentBricks = [];
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks,
            });
            iframe.handleClick();
            expect(mockBlock.manageActions).toHaveBeenCalledWith([
                ...parentBricks,
                iframe,
            ]);
        });
        it("should handle click without event object", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => iframe.handleClick()).not.toThrow();
            expect(mockBlock.manageActions).toHaveBeenCalled();
        });
    });
    describe("integration scenarios", () => {
        it("should complete activation-deactivation cycle", () => {
            const existingIframe = document.createElement("iframe");
            existingIframe.src = "https://example.com";
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.activate();
            expect(element.style.position).toBe("relative");
            expect(element.contains(iframe.mask)).toBe(true);
            iframe.desactivate();
            expect(element.hasAttribute("style")).toBe(false);
            expect(element.contains(iframe.mask)).toBe(false);
        });
        it("should handle multiple activate calls", () => {
            const existingIframe = document.createElement("iframe");
            element.appendChild(existingIframe);
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.activate();
            iframe.activate();
            const masks = element.querySelectorAll("div");
            const maskCount = Array.from(masks).filter((div) => div.style.position === "absolute").length;
            expect(maskCount).toBe(1);
        });
        it("should update iframe dimensions multiple times", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com", "800", "600", false);
            let createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("width")).toBe("800");
            iframe.updateIframe("https://example.com", "1200", "900", true);
            createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("width")).toBe("1200");
            expect(createdIframe?.getAttribute("height")).toBe("900");
            expect(createdIframe?.getAttribute("allow")).toBe("fullscreen");
        });
        it("should handle workflow: create, activate, update, deactivate", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            parseUrlMock.mockReturnValue({
                url: "https://example.com/video",
                host: "example.com",
            });
            iframe.updateIframe("https://example.com/video", "800", "600", false);
            expect(element.querySelector("iframe")).toBeTruthy();
            iframe.activate();
            expect(element.style.position).toBe("relative");
            parseUrlMock.mockReturnValue({
                url: "https://example.com/video2",
                host: "example.com",
            });
            iframe.updateIframe("https://example.com/video2", "1000", "700", true);
            const updatedIframe = element.querySelector("iframe");
            expect(updatedIframe?.getAttribute("src")).toBe("https://example.com/video2");
            iframe.desactivate();
            expect(element.hasAttribute("style")).toBe(false);
        });
    });
    describe("edge cases", () => {
        it("should handle empty strings for dimensions", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com", "", "", false);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("width")).toBe("");
            expect(createdIframe?.getAttribute("height")).toBe("");
        });
        it("should handle special characters in URL", () => {
            parseUrlMock.mockReturnValue({
                url: "https://example.com/video?param=value&other=test",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com/video?param=value&other=test", "800", "600", false);
            const createdIframe = element.querySelector("iframe");
            expect(createdIframe?.getAttribute("src")).toBe("https://example.com/video?param=value&other=test");
        });
        it("should preserve element children other than iframe and mask", () => {
            const otherChild = document.createElement("span");
            otherChild.textContent = "Other content";
            element.appendChild(otherChild);
            parseUrlMock.mockReturnValue({
                url: "https://example.com",
                host: "example.com",
            });
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            iframe.updateIframe("https://example.com", "800", "600", false);
            expect(element.querySelector("span")).toBe(otherChild);
        });
        it("should handle mask removal when not in DOM", () => {
            const iframe = new Iframe({
                block: mockBlock,
                element,
                parentBricks: [],
            });
            expect(() => iframe.desactivate()).not.toThrow();
        });
    });
});
