import Brick from "../core/brick.js";
import { parseUrl } from "../services/url.js";
import YoutubePlaylistModal from "../user-interface/plugin/youtube-playlist-modal.js";
import { snackbar } from "../user-interface/common/eden-snackbar.js";
import html from "../services/html.js";
import t from "../i18n/index.js";
const SUPPORTED_HOSTS = {
    "www.youtube.com": true,
    "www.youtube-nocookie.com": true,
};
export default class YoutubePlaylist extends Brick {
    constructor({ block, element, parentBricks }) {
        super({ block, element, parentBricks });
        this.updateUrl = (url) => {
            const parsedUrl = parseUrl(url);
            if (!parsedUrl?.host) {
                snackbar(t("snackbars.fill-valid-url"), "error");
                return;
            }
            const embedUrl = this.getEmbedUrl(parsedUrl);
            if (embedUrl === undefined) {
                snackbar(t("snackbars.video-service-not-supported"), "error");
                return;
            }
            this.updateIframeOrCreate(embedUrl);
        };
        this.activate = () => {
            if (!this.element)
                return;
            this.element.addEventListener("click", this.handleClick);
            const iframe = this.element.querySelector("iframe");
            if (iframe) {
                this.element.appendChild(this.mask);
            }
        };
        this.desactivate = () => {
            if (!this.element)
                return;
            this.element.removeEventListener("click", this.handleClick);
            this.mask.remove();
        };
        this.mask = this.createMaskElement();
        this.actions = [this.createPlaylistAction()];
    }
    createMaskElement() {
        return Object.assign(document.createElement("div"), {
            style: `
        position: absolute;
        top: 0;
        bottom: 0;
        left: 0;
        right: 0;
      `,
        });
    }
    createPlaylistAction() {
        return {
            tooltip: t("bricks.youtube-playlist.add-playlist"),
            icon: "movie",
            action: () => {
                const iframe = this.element.querySelector("iframe");
                const videoUrl = iframe?.getAttribute("src") ?? "";
                new YoutubePlaylistModal(this.updateUrl, videoUrl);
            },
        };
    }
    isSupportedHost(host) {
        return host in SUPPORTED_HOSTS;
    }
    getEmbedUrl(parsedUrl) {
        if (!parsedUrl || !this.isSupportedHost(parsedUrl.host)) {
            return undefined;
        }
        return parsedUrl.url;
    }
    updateIframeOrCreate(embedUrl) {
        const iframe = this.element.querySelector("iframe");
        if (iframe) {
            iframe.src = embedUrl;
        }
        else {
            this.element.innerHTML = html `<iframe
        src="${embedUrl}"
        frameborder="0"
        webkitAllowFullScreen
        mozallowfullscreen
        allowfullscreen
      ></iframe>`;
            this.element.appendChild(this.mask);
        }
    }
}
