import { describe, it, expect, beforeEach, afterEach, vi } from "vitest";
import Eden from "../eden";
import { EdenMode } from "../types";
import { setCSSTranslations } from "../services/css";
vi.mock("../user-interface/common/eden-snackbar.js", () => ({
    snackbar: vi.fn(),
}));
vi.mock("../services/fetcher.js", () => ({
    fetchFormats: vi.fn(() => Promise.resolve("{}")),
}));
vi.mock("../services/util.js", () => ({
    disableBrowserTranslation: vi.fn(),
    isLocalhost: vi.fn(),
}));
vi.mock("../services/cleanup.js", () => ({
    closeDetailsElements: vi.fn(),
    cleanupTagsAndPreserveContent: vi.fn(),
    randomizeSvgSpriteCalls: vi.fn(),
    cleanupRandomSvgSpriteCallParams: vi.fn(),
    cleanupOldSvgAttributes: vi.fn(),
}));
vi.mock("../services/api.js", () => ({
    checkLastUpdated: vi.fn(() => Promise.resolve({ lastUpdated: null, lastUpdatedBy: null })),
    saveZones: vi.fn(() => Promise.resolve({ error: "" })),
}));
vi.mock("../services/token.js", () => ({
    decodeToken: vi.fn(() => ({
        userId: "test-user",
        role: "admin",
    })),
}));
vi.mock("../services/event.js", () => ({
    fireEvent: vi.fn(),
}));
vi.mock("../services/templates.js", () => ({
    fillTemplates: vi.fn((templates) => templates),
    groupTemplates: vi.fn((templates) => ({})),
}));
vi.mock("../i18n/index.js", () => ({
    default: vi.fn((key) => {
        const translations = {
            "ui.css-labels.description": "Description",
            "ui.css-labels.preview": "Preview",
            "ui.css-labels.no-results": "No Results",
            "ui.css-labels.scheduled": "Scheduled",
            "ui.css-labels.active": "Active",
            "ui.css-labels.disabled": "Disabled",
            "ui.css-labels.before": "Before",
            "ui.css-labels.inside": "Inside",
            "ui.css-labels.after": "After",
            "snackbars.content-saved": "Content saved",
        };
        return translations[key] || key;
    }),
}));
vi.mock("../core/zone.js", () => ({
    default: class MockZone {
        constructor(element, descriptors) {
            this.element = element;
            this.descriptors = descriptors;
        }
        async analyse() { }
        activate() { }
        desactivate() { }
        preSave() { }
        postSave() { }
        displayInfo() { }
        hideInfo() { }
        cleanRefs() { }
    },
}));
vi.mock("../core/template.js", () => ({
    default: class MockTemplate {
        constructor(template) {
            this.template = template;
        }
    },
}));
vi.mock("../user-interface/page/eden-main-ui.js", () => ({
    default: class MockEdenMainUi {
        toggleActive() { }
        startLoading() { }
        stopLoading() { }
        remove() { }
    },
}));
vi.mock("../user-interface/icons/eden-icon.js");
vi.mock("../user-interface/common/eden-button.js");
vi.mock("../user-interface/common/eden-checkbox.js");
vi.mock("../user-interface/common/eden-input.js");
vi.mock("../user-interface/common/eden-tag-manager.js");
vi.mock("../user-interface/plugin/last-updated-warning-modal.js", () => ({
    default: class MockWarningModal {
        constructor(onConfirm, lastUpdated, lastUpdatedBy) {
            this.onConfirm = onConfirm;
            this.lastUpdated = lastUpdated;
            this.lastUpdatedBy = lastUpdatedBy;
        }
        init() { }
    },
}));
vi.mock("../user-interface/plugin/edit-structure-modal.js", () => ({
    default: class MockEditStructureModal {
        init() { }
    },
}));
describe("Eden Class", () => {
    let eden;
    beforeEach(() => {
        document.body.innerHTML = "";
        if (document.documentElement) {
            const attrs = document.documentElement.attributes;
            for (let i = attrs.length - 1; i >= 0; i--) {
                document.documentElement.removeAttribute(attrs[i].name);
            }
            document.documentElement.setAttribute("style", "");
        }
        window.sessionStorage.clear();
        window.EdenConfig = { lang: "en" };
        window.updateEdenDraftMode = undefined;
        window.setEdenMode = undefined;
        eden = new Eden();
    });
    afterEach(() => {
        vi.clearAllMocks();
    });
    describe("constructor & properties", () => {
        it("should initialize with empty arrays and objects", () => {
            const instance = new Eden();
            expect(instance.zones).toBeUndefined();
            expect(instance.templateDescriptors).toBeUndefined();
            expect(instance.activeBlock).toBeUndefined();
        });
    });
    describe("init()", () => {
        it("should initialize zones and templateDescriptors as empty", () => {
            eden.init();
            expect(eden.zones).toEqual([]);
            expect(eden.templateDescriptors).toEqual({});
        });
        it("should decode user token", () => {
            eden.init();
            expect(eden.user).toEqual({ userId: "test-user", role: "admin" });
        });
        it("should set CSS custom properties for translations", () => {
            eden.init();
            const root = document.documentElement;
            expect(root.style.getPropertyValue("--eden-label-description")).toBe('"Description"');
            expect(root.style.getPropertyValue("--eden-label-active")).toBe('"Active"');
        });
        it("should call cleanup functions", async () => {
            const { cleanupOldSvgAttributes, randomizeSvgSpriteCalls } = await import("../services/cleanup.js");
            eden.init();
            expect(cleanupOldSvgAttributes).toHaveBeenCalled();
            expect(randomizeSvgSpriteCalls).toHaveBeenCalled();
        });
    });
    describe("setCSSTranslations()", () => {
        it("should set all CSS label properties", () => {
            setCSSTranslations();
            const root = document.documentElement;
            const labels = [
                "description",
                "preview",
                "no-results",
                "scheduled",
                "active",
                "disabled",
                "before",
                "inside",
                "after",
            ];
            labels.forEach((label) => {
                const value = root.style.getPropertyValue(`--eden-label-${label}`);
                expect(value).toBeTruthy();
                expect(value).toContain('"');
            });
        });
        it("should use i18n translations", () => {
            setCSSTranslations();
            const root = document.documentElement;
            expect(root.style.getPropertyValue("--eden-label-description")).toBe('"Description"');
        });
    });
    describe("start()", () => {
        it("should initialize and change mode to connected", async () => {
            const changeModeSpy = vi.spyOn(eden, "changeMode");
            const initSpy = vi.spyOn(eden, "init");
            await eden.start();
            expect(initSpy).toHaveBeenCalled();
            expect(changeModeSpy).toHaveBeenCalledWith(EdenMode.connected);
        });
        it("should register templates", async () => {
            const registerSpy = vi.spyOn(eden, "register");
            await eden.start();
            expect(registerSpy).toHaveBeenCalled();
        });
        it("should analyse zones", async () => {
            const analyseSpy = vi.spyOn(eden, "analyse");
            await eden.start();
            expect(analyseSpy).toHaveBeenCalled();
        });
        it("should create and append EdenPageUi", async () => {
            const appendSpy = vi.spyOn(document.body, "append");
            await eden.start();
            expect(eden.pageUi).toBeDefined();
            expect(appendSpy).toHaveBeenCalled();
        });
        it("should set startedAt timestamp", async () => {
            await eden.start();
            expect(eden.startedAt).toBeGreaterThan(0);
        });
    });
    describe("register()", () => {
        beforeEach(() => {
            eden.init();
        });
        it("should fetch and fill templates", async () => {
            const { fetchFormats } = await import("../services/fetcher.js");
            await eden.register();
            expect(fetchFormats).toHaveBeenCalled();
        });
        it("should create template descriptors", async () => {
            await eden.register();
            expect(eden.templateDescriptors).toBeDefined();
        });
        it("should set templates and groupedTemplates", async () => {
            await eden.register();
            expect(eden.templates).toBeDefined();
            expect(eden.groupedTemplates).toBeDefined();
        });
    });
    describe("analyse()", () => {
        beforeEach(() => {
            eden.init();
        });
        it("should find and analyse all zone elements", async () => {
            const zone1 = document.createElement("div");
            zone1.setAttribute("data-eden-zone", "true");
            const zone2 = document.createElement("div");
            zone2.setAttribute("data-eden-zone", "true");
            document.body.append(zone1, zone2);
            await eden.analyse();
            expect(eden.zones).toHaveLength(2);
        });
        it("should handle no zone elements", async () => {
            await eden.analyse();
            expect(eden.zones).toEqual([]);
        });
        it("should call analyse on each zone", async () => {
            const zone1 = document.createElement("div");
            zone1.setAttribute("data-eden-zone", "true");
            document.body.append(zone1);
            await eden.analyse();
            expect(eden.zones[0]).toBeDefined();
        });
    });
    describe("addZone()", () => {
        beforeEach(() => {
            eden.init();
        });
        it("should add new zone and analyse it", async () => {
            const zoneElement = document.createElement("div");
            zoneElement.setAttribute("data-eden-zone", "true");
            await eden.addZone(zoneElement);
            expect(eden.zones).toHaveLength(1);
        });
        it("should return the created zone", async () => {
            const zoneElement = document.createElement("div");
            zoneElement.setAttribute("data-eden-zone", "true");
            const zone = await eden.addZone(zoneElement);
            expect(zone).toBeDefined();
        });
        it("should add zone to zones array", async () => {
            const zone1 = document.createElement("div");
            zone1.setAttribute("data-eden-zone", "true");
            const zone2 = document.createElement("div");
            zone2.setAttribute("data-eden-zone", "true");
            await eden.addZone(zone1);
            expect(eden.zones).toHaveLength(1);
            await eden.addZone(zone2);
            expect(eden.zones).toHaveLength(2);
        });
    });
    describe("stop()", () => {
        beforeEach(async () => {
            await eden.start();
        });
        it("should desactivate zones", () => {
            const desactivateSpy = vi.spyOn(eden, "desactivate");
            eden.stop();
            expect(desactivateSpy).toHaveBeenCalled();
        });
        it("should remove pageUi", () => {
            const removeSpy = vi.spyOn(eden.pageUi, "remove");
            eden.stop();
            expect(removeSpy).toHaveBeenCalled();
        });
        it("should clean refs", () => {
            const cleanRefsSpy = vi.spyOn(eden, "cleanRefs");
            eden.stop();
            expect(cleanRefsSpy).toHaveBeenCalled();
        });
        it("should reinitialize", () => {
            const initSpy = vi.spyOn(eden, "init");
            eden.stop();
            expect(initSpy).toHaveBeenCalled();
        });
    });
    describe("cleanRefs()", () => {
        beforeEach(async () => {
            const zone1 = document.createElement("div");
            zone1.setAttribute("data-eden-zone", "true");
            document.body.append(zone1);
            eden.init();
            await eden.analyse();
        });
        it("should call cleanRefs on all zones", () => {
            const cleanRefsSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].cleanRefs = cleanRefsSpy;
            }
            eden.cleanRefs();
            if (eden.zones[0]) {
                expect(cleanRefsSpy).toHaveBeenCalled();
            }
        });
    });
    describe("activate()", () => {
        beforeEach(async () => {
            eden.init();
            await eden.analyse();
        });
        it("should activate all zones", () => {
            const activateSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].activate = activateSpy;
            }
            eden.activate();
            if (eden.zones[0]) {
                expect(activateSpy).toHaveBeenCalled();
            }
        });
        it("should change mode to editing", () => {
            const changeModeSpy = vi.spyOn(eden, "changeMode");
            eden.activate();
            expect(changeModeSpy).toHaveBeenCalledWith(EdenMode.editing);
        });
    });
    describe("desactivate()", () => {
        beforeEach(async () => {
            eden.init();
            await eden.analyse();
        });
        it("should desactivate all zones", () => {
            const desactivateSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].desactivate = desactivateSpy;
            }
            eden.desactivate();
            if (eden.zones[0]) {
                expect(desactivateSpy).toHaveBeenCalled();
            }
        });
        it("should change mode to connected", () => {
            const changeModeSpy = vi.spyOn(eden, "changeMode");
            eden.desactivate();
            expect(changeModeSpy).toHaveBeenCalledWith(EdenMode.connected);
        });
    });
    describe("preSave()", () => {
        beforeEach(async () => {
            eden.init();
            await eden.analyse();
        });
        it("should call preSave on all zones", () => {
            const preSaveSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].preSave = preSaveSpy;
            }
            eden.preSave();
            if (eden.zones[0]) {
                expect(preSaveSpy).toHaveBeenCalled();
            }
        });
    });
    describe("save()", () => {
        beforeEach(async () => {
            window.EdenConfig = { lang: "en" };
            await eden.start();
        });
        it("should desactivate zones before saving", async () => {
            const desactivateSpy = vi.spyOn(eden, "desactivate");
            await eden.save();
            expect(desactivateSpy).toHaveBeenCalled();
        });
        it("should call preSave", async () => {
            const preSaveSpy = vi.spyOn(eden, "preSave");
            await eden.save();
            expect(preSaveSpy).toHaveBeenCalled();
        });
        it("should clean markup", async () => {
            const cleanMarkupSpy = vi.spyOn(eden, "cleanMarkup");
            await eden.save();
            expect(cleanMarkupSpy).toHaveBeenCalled();
        });
    });
    describe("postSave()", () => {
        beforeEach(async () => {
            await eden.start();
        });
        it("should show error snackbar if error exists", async () => {
            const { snackbar } = await import("../user-interface/common/eden-snackbar.js");
            eden.postSave("Test error");
            expect(snackbar).toHaveBeenCalledWith("Test error", "error");
        });
        it("should call postSave on all zones if no error", async () => {
            const postSaveSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].postSave = postSaveSpy;
            }
            eden.postSave("");
            if (eden.zones[0]) {
                expect(postSaveSpy).toHaveBeenCalled();
            }
        });
        it("should fire eden-content-saved event if no error", async () => {
            const { fireEvent } = await import("../services/event.js");
            eden.postSave("");
            expect(fireEvent).toHaveBeenCalledWith(document, "eden-content-saved");
        });
        it("should show success snackbar if no error", async () => {
            const { snackbar } = await import("../user-interface/common/eden-snackbar.js");
            eden.postSave("");
            expect(snackbar).toHaveBeenCalledWith("Content saved", "success");
        });
        it("should update startedAt timestamp", async () => {
            const oldStartTime = eden.startedAt;
            eden.postSave("");
            await new Promise((resolve) => setTimeout(resolve, 150));
            expect(eden.startedAt).toBeGreaterThan(oldStartTime);
        });
    });
    describe("checkLastUpdated()", () => {
        beforeEach(async () => {
            window.EdenConfig = { lang: "en" };
            await eden.start();
        });
    });
    describe("cleanMarkup()", () => {
        it("should remove unwanted elements", () => {
            const grammarly = document.createElement("grammarly-extension");
            const ltHighlighter = document.createElement("lt-highlighter");
            const edenZoneUi = document.createElement("eden-zone-ui");
            document.body.append(grammarly, ltHighlighter, edenZoneUi);
            eden.cleanMarkup();
            expect(document.body.contains(grammarly)).toBe(false);
            expect(document.body.contains(ltHighlighter)).toBe(false);
            expect(document.body.contains(edenZoneUi)).toBe(false);
        });
        it("should remove unwanted attributes", () => {
            const element = document.createElement("div");
            element.setAttribute("spellcheck", "true");
            element.setAttribute("data-gramm", "true");
            element.setAttribute("data-eden-helper", "true");
            document.body.append(element);
            eden.cleanMarkup();
            expect(element.hasAttribute("spellcheck")).toBe(false);
            expect(element.hasAttribute("data-gramm")).toBe(false);
            expect(element.hasAttribute("data-eden-helper")).toBe(false);
        });
        it("should remove undefined classes", () => {
            const element = document.createElement("div");
            element.classList.add("undefined");
            element.classList.add("valid-class");
            document.body.append(element);
            eden.cleanMarkup();
            expect(element.classList.contains("undefined")).toBe(false);
            expect(element.classList.contains("valid-class")).toBe(true);
        });
        it("should call cleanup service functions", async () => {
            const { cleanupTagsAndPreserveContent, closeDetailsElements, cleanupRandomSvgSpriteCallParams, } = await import("../services/cleanup.js");
            eden.cleanMarkup();
            expect(cleanupTagsAndPreserveContent).toHaveBeenCalled();
            expect(closeDetailsElements).toHaveBeenCalled();
            expect(cleanupRandomSvgSpriteCallParams).toHaveBeenCalled();
        });
    });
    describe("getActiveBlock() & setActiveBlock()", () => {
        it("should get active block", () => {
            const mockBlock = {};
            eden.setActiveBlock(mockBlock);
            const activeBlock = eden.getActiveBlock();
            expect(activeBlock).toBe(mockBlock);
        });
        it("should set active block to undefined", () => {
            const mockBlock = {};
            eden.setActiveBlock(mockBlock);
            eden.setActiveBlock(undefined);
            const activeBlock = eden.getActiveBlock();
            expect(activeBlock).toBeUndefined();
        });
    });
    describe("logout()", () => {
        beforeEach(async () => {
            window.sessionStorage.setItem("eden_token", "test-token");
            await eden.start();
        });
        it("should call updateEdenDraftMode if it exists", async () => {
            const mockUpdateDraftMode = vi.fn();
            window.updateEdenDraftMode = mockUpdateDraftMode;
            await eden.logout();
            expect(mockUpdateDraftMode).toHaveBeenCalledWith(false, "test-token");
        });
        it("should desactivate zones", async () => {
            const desactivateSpy = vi.spyOn(eden, "desactivate");
            await eden.logout();
            expect(desactivateSpy).toHaveBeenCalled();
        });
        it("should remove pageUi", async () => {
            const removeSpy = vi.spyOn(eden.pageUi, "remove");
            await eden.logout();
            expect(removeSpy).toHaveBeenCalled();
        });
        it("should remove token from session storage", async () => {
            window.sessionStorage.setItem("eden_token", "test-token");
            await eden.logout();
            expect(window.sessionStorage.getItem("eden_token")).toBeNull();
        });
        it("should reload window", async () => {
            const reloadSpy = vi.spyOn(window.location, "reload");
            await eden.logout();
            expect(reloadSpy).toHaveBeenCalled();
        });
    });
    describe("displayInfo()", () => {
        beforeEach(async () => {
            eden.init();
            await eden.analyse();
        });
        it("should display info on all zones", () => {
            const displayInfoSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].displayInfo = displayInfoSpy;
            }
            eden.displayInfo();
            if (eden.zones[0]) {
                expect(displayInfoSpy).toHaveBeenCalled();
            }
        });
        it("should change mode to info", () => {
            const changeModeSpy = vi.spyOn(eden, "changeMode");
            eden.displayInfo();
            expect(changeModeSpy).toHaveBeenCalledWith(EdenMode.info);
        });
    });
    describe("hideInfo()", () => {
        beforeEach(async () => {
            eden.init();
            await eden.analyse();
        });
        it("should hide info on all zones", () => {
            const hideInfoSpy = vi.fn();
            if (eden.zones[0]) {
                eden.zones[0].hideInfo = hideInfoSpy;
            }
            eden.hideInfo();
            if (eden.zones[0]) {
                expect(hideInfoSpy).toHaveBeenCalled();
            }
        });
        it("should change mode to connected", () => {
            const changeModeSpy = vi.spyOn(eden, "changeMode");
            eden.hideInfo();
            expect(changeModeSpy).toHaveBeenCalledWith(EdenMode.connected);
        });
    });
    describe("changeMode()", () => {
        it("should set data-eden-mode attribute", () => {
            eden.changeMode(EdenMode.editing);
            expect(document.documentElement.getAttribute("data-eden-mode")).toBe(EdenMode.editing);
        });
        it("should store old mode in session storage", () => {
            eden.changeMode(EdenMode.editing);
            expect(window.sessionStorage.getItem("old-eden-mode")).toBe(EdenMode.editing);
        });
        it("should remove old mode before setting new one", () => {
            eden.changeMode(EdenMode.editing);
            eden.changeMode(EdenMode.connected);
            expect(window.sessionStorage.getItem("old-eden-mode")).toBe(EdenMode.connected);
        });
        it("should call window.setEdenMode if it exists", () => {
            const mockSetMode = vi.fn();
            window.setEdenMode = mockSetMode;
            eden.changeMode(EdenMode.editing);
            expect(mockSetMode).toHaveBeenCalledWith(EdenMode.editing);
        });
        it("should handle null/falsy mode", () => {
            eden.changeMode(EdenMode.editing);
            eden.changeMode(null);
            expect(window.sessionStorage.getItem("old-eden-mode")).toBeNull();
        });
    });
    describe("integration scenarios", () => {
        it("should handle complete editing workflow", async () => {
            window.EdenConfig = { lang: "en" };
            await eden.start();
            expect(eden.zones).toBeDefined();
            eden.activate();
            expect(document.documentElement.getAttribute("data-eden-mode")).toBe(EdenMode.editing);
            await eden.save();
            expect(document.documentElement.getAttribute("data-eden-mode")).toBe(EdenMode.connected);
        });
        it("should handle stop and restart", async () => {
            window.EdenConfig = { lang: "en" };
            await eden.start();
            const startedAt = eden.startedAt;
            eden.stop();
            expect(eden.zones).toEqual([]);
            await eden.start();
            expect(eden.zones).toBeDefined();
            expect(eden.startedAt).toBeGreaterThanOrEqual(startedAt);
        });
    });
});
